/***************************************************************************
                        qgsmulticurve.h
  -------------------------------------------------------------------
Date                 : 28 Oct 2014
Copyright            : (C) 2014 by Marco Hugentobler
email                : marco.hugentobler at sourcepole dot com
 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef QGSMULTICURVE_H
#define QGSMULTICURVE_H

#include "qgis_core.h"
#include "qgis_sip.h"
#include "qgsgeometrycollection.h"

#include <QString>

using namespace Qt::StringLiterals;

/**
 * \ingroup core
 * \class QgsMultiCurve
 * \brief Multi curve geometry collection.
 */
class CORE_EXPORT QgsMultiCurve: public QgsGeometryCollection
{
  public:
    QgsMultiCurve();


#ifndef SIP_RUN

    /**
     * Returns the curve with the specified \a index.
     *
     * \since QGIS 3.16
     */
    QgsCurve *curveN( int index );
#else

    /**
     * Returns the curve with the specified \a index.
     *
     * \throws IndexError if no curve with the specified index exists.
     *
     * \since QGIS 3.16
     */
    SIP_PYOBJECT curveN( int index ) SIP_TYPEHINT( QgsCurve );
    % MethodCode
    if ( a0 < 0 || a0 >= sipCpp->numGeometries() )
    {
      PyErr_SetString( PyExc_IndexError, QByteArray::number( a0 ) );
      sipIsErr = 1;
    }
    else
    {
      return sipConvertFromType( sipCpp->curveN( a0 ), sipType_QgsCurve, NULL );
    }
    % End
#endif

#ifndef SIP_RUN

    /**
     * Returns the curve with the specified \a index.
     *
     * \note Not available in Python bindings
     *
     * \since QGIS 3.16
     */
    const QgsCurve *curveN( int index ) const;
#endif

    QString geometryType() const override SIP_HOLDGIL;
    QgsMultiCurve *clone() const override SIP_FACTORY;
    void clear() override;
    QgsMultiCurve *toCurveType() const override SIP_FACTORY;
    bool fromWkt( const QString &wkt ) override;
    QDomElement asGml2( QDomDocument &doc, int precision = 17, const QString &ns = "gml", QgsAbstractGeometry::AxisOrder axisOrder = QgsAbstractGeometry::AxisOrder::XY ) const override;
    QDomElement asGml3( QDomDocument &doc, int precision = 17, const QString &ns = "gml", QgsAbstractGeometry::AxisOrder axisOrder = QgsAbstractGeometry::AxisOrder::XY ) const override;
    json asJsonObject( int precision = 17 ) const override SIP_SKIP;
    bool addGeometry( QgsAbstractGeometry *g SIP_TRANSFER ) override;
    bool addGeometries( const QVector< QgsAbstractGeometry * > &geometries SIP_TRANSFER ) override;
    bool insertGeometry( QgsAbstractGeometry *g SIP_TRANSFER, int index ) override;
    QgsMultiCurve *simplifyByDistance( double tolerance ) const override SIP_FACTORY;

    /**
     * Returns a copy of the multi curve, where each component curve has had its line direction reversed.
     */
    QgsMultiCurve *reversed() const SIP_FACTORY;

    QgsAbstractGeometry *boundary() const override SIP_FACTORY;

#ifndef SIP_RUN

    /**
     * Cast the \a geom to a QgsMultiCurve.
     * Should be used by qgsgeometry_cast<QgsMultiCurve *>( geometry ).
     *
     * Objects will be automatically converted to the appropriate target type.
     *
     * \note Not available in Python.
     */
    inline static const QgsMultiCurve *cast( const QgsAbstractGeometry *geom ) // cppcheck-suppress duplInheritedMember
    {
      if ( !geom )
        return nullptr;

      const Qgis::WkbType flatType = QgsWkbTypes::flatType( geom->wkbType() );
      if ( flatType == Qgis::WkbType::MultiCurve
           || flatType == Qgis::WkbType::MultiLineString )
        return static_cast<const QgsMultiCurve *>( geom );
      return nullptr;
    }

    /**
     * Cast the \a geom to a QgsMultiCurve.
     * Should be used by qgsgeometry_cast<QgsMultiCurve *>( geometry ).
     *
     * Objects will be automatically converted to the appropriate target type.
     *
     * \note Not available in Python.
     */
    inline static QgsMultiCurve *cast( QgsAbstractGeometry *geom ) // cppcheck-suppress duplInheritedMember
    {
      if ( !geom )
        return nullptr;

      const Qgis::WkbType flatType = QgsWkbTypes::flatType( geom->wkbType() );
      if ( flatType == Qgis::WkbType::MultiCurve
           || flatType == Qgis::WkbType::MultiLineString )
        return static_cast<QgsMultiCurve *>( geom );
      return nullptr;
    }
#endif

    QgsMultiCurve *createEmptyWithSameType() const override SIP_FACTORY;

#ifdef SIP_RUN
    SIP_PYOBJECT __repr__();
    % MethodCode
    QString wkt = sipCpp->asWkt();
    if ( wkt.length() > 1000 )
      wkt = wkt.left( 1000 ) + u"..."_s;
    QString str = u"<QgsMultiCurve: %1>"_s.arg( wkt );
    sipRes = PyUnicode_FromString( str.toUtf8().constData() );
    % End
#endif

};

// clazy:excludeall=qstring-allocations

#endif // QGSMULTICURVE_H
