---
name: t.rast.accumulate.py
description: Computes cyclic accumulations of a space time raster dataset.
keywords: [ temporal, accumulation, raster, time ]
---

# t.rast.accumulate.py

Computes cyclic accumulations of a space time raster dataset.

=== "Command line"

    **t.rast.accumulate.py**
    [**-nr**]
    **input**=*name*
    **output**=*name*
    [**lower**=*name*]
    [**upper**=*name*]
    **start**=*string*
    [**stop**=*string*]
    **cycle**=*string*
    [**offset**=*string*]
    [**granularity**=*string*]
    **basename**=*string*
    [**suffix**=*name*]
    **limits**=*lower,upper*
    [**scale**=*float*]
    [**shift**=*float*]
    [**method**=*string*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    t.rast.accumulate.py input=name output=name start=string cycle=string basename=string limits=0.0
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***t.rast.accumulate.py***",
        **input**,
        **output**,
        **lower**=*None*,
        **upper**=*None*,
        **start**,
        **stop**=*None*,
        **cycle**,
        **offset**=*None*,
        **granularity**=*"1 day"*,
        **basename**,
        **suffix**=*"gran"*,
        **limits**,
        **scale**=*None*,
        **shift**=*None*,
        **method**=*"mean"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("t.rast.accumulate.py", input="name", output="name", start="string", cycle="string", basename="string", limits=0.0)
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.t_rast_accumulate_py*(**input**,
        **output**,
        **lower**=*None*,
        **upper**=*None*,
        **start**,
        **stop**=*None*,
        **cycle**,
        **offset**=*None*,
        **granularity**=*"1 day"*,
        **basename**,
        **suffix**=*"gran"*,
        **limits**,
        **scale**=*None*,
        **shift**=*None*,
        **method**=*"mean"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.t_rast_accumulate_py(input="name", output="name", start="string", cycle="string", basename="string", limits=0.0)
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the input space time raster dataset  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    **lower**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input space time raster dataset that defines the lower threshold, values lower than this threshold are excluded from accumulation  
    **upper**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input space time raster dataset that defines the upper threshold, values higher than this threshold are excluded from accumulation  
    **start**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;The temporal starting point to begin the accumulation, eg '2001-01-01'  
    **stop**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;The temporal date to stop the accumulation, eg '2009-01-01'  
    **cycle**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;The temporal cycle to restart the accumulation, eg '12 months'  
    **offset**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;The temporal offset to the beginning of the next cycle, eg '6 months'  
    **granularity**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;The granularity for accumulation '1 day'  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1 day*  
    **basename**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename of the new generated output maps  
    &nbsp;&nbsp;&nbsp;&nbsp;A numerical suffix separated by an underscore will be attached to create a unique identifier  
    **suffix**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Suffix to add at basename: set 'gran' for granularity, 'time' for the full time format, 'num' for numerical suffix with a specific number of digits (default %%05)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *gran*  
    **limits**=*lower,upper* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use these limits in case lower and/or upper input space time raster datasets are not defined or contain NULL values  
    **scale**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Scale factor for input space time raster dataset  
    **shift**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Shift factor for input space time raster dataset  
    **method**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;This method will be applied to compute the accumulative values from the input maps in a single granule  
    &nbsp;&nbsp;&nbsp;&nbsp;Growing Degree Days or Winkler indices; Mean: sum(input maps)/(number of input maps); Biologically Effective Degree Days; Huglin Heliothermal index  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *mean, gdd, bedd, huglin*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *mean*  
    **-n**  
    &nbsp;&nbsp;&nbsp;&nbsp;Register empty maps in the output space time raster dataset, otherwise they will be deleted  
    **-r**  
    &nbsp;&nbsp;&nbsp;&nbsp;Reverse time direction in cyclic accumulation  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the input space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, strds, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, strds, *name*  
    **lower** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input space time raster dataset that defines the lower threshold, values lower than this threshold are excluded from accumulation  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, strds, *name*  
    **upper** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input space time raster dataset that defines the upper threshold, values higher than this threshold are excluded from accumulation  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, strds, *name*  
    **start** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The temporal starting point to begin the accumulation, eg '2001-01-01'  
    **stop** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The temporal date to stop the accumulation, eg '2009-01-01'  
    **cycle** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The temporal cycle to restart the accumulation, eg '12 months'  
    **offset** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The temporal offset to the beginning of the next cycle, eg '6 months'  
    **granularity** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The granularity for accumulation '1 day'  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1 day*  
    **basename** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename of the new generated output maps  
    &nbsp;&nbsp;&nbsp;&nbsp;A numerical suffix separated by an underscore will be attached to create a unique identifier  
    **suffix** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Suffix to add at basename: set 'gran' for granularity, 'time' for the full time format, 'num' for numerical suffix with a specific number of digits (default %%05)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *gran*  
    **limits** : tuple[float, float] | list[float] | str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Use these limits in case lower and/or upper input space time raster datasets are not defined or contain NULL values  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *lower,upper*  
    **scale** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Scale factor for input space time raster dataset  
    **shift** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Shift factor for input space time raster dataset  
    **method** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;This method will be applied to compute the accumulative values from the input maps in a single granule  
    &nbsp;&nbsp;&nbsp;&nbsp;Growing Degree Days or Winkler indices; Mean: sum(input maps)/(number of input maps); Biologically Effective Degree Days; Huglin Heliothermal index  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *mean, gdd, bedd, huglin*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *mean*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *n*, *r*  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Register empty maps in the output space time raster dataset, otherwise they will be deleted  
    &nbsp;&nbsp;&nbsp;&nbsp;**r**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Reverse time direction in cyclic accumulation  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the input space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, strds, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, strds, *name*  
    **lower** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input space time raster dataset that defines the lower threshold, values lower than this threshold are excluded from accumulation  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, strds, *name*  
    **upper** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input space time raster dataset that defines the upper threshold, values higher than this threshold are excluded from accumulation  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, strds, *name*  
    **start** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The temporal starting point to begin the accumulation, eg '2001-01-01'  
    **stop** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The temporal date to stop the accumulation, eg '2009-01-01'  
    **cycle** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The temporal cycle to restart the accumulation, eg '12 months'  
    **offset** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The temporal offset to the beginning of the next cycle, eg '6 months'  
    **granularity** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The granularity for accumulation '1 day'  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1 day*  
    **basename** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename of the new generated output maps  
    &nbsp;&nbsp;&nbsp;&nbsp;A numerical suffix separated by an underscore will be attached to create a unique identifier  
    **suffix** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Suffix to add at basename: set 'gran' for granularity, 'time' for the full time format, 'num' for numerical suffix with a specific number of digits (default %%05)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *gran*  
    **limits** : tuple[float, float] | list[float] | str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Use these limits in case lower and/or upper input space time raster datasets are not defined or contain NULL values  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *lower,upper*  
    **scale** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Scale factor for input space time raster dataset  
    **shift** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Shift factor for input space time raster dataset  
    **method** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;This method will be applied to compute the accumulative values from the input maps in a single granule  
    &nbsp;&nbsp;&nbsp;&nbsp;Growing Degree Days or Winkler indices; Mean: sum(input maps)/(number of input maps); Biologically Effective Degree Days; Huglin Heliothermal index  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *mean, gdd, bedd, huglin*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *mean*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *n*, *r*  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Register empty maps in the output space time raster dataset, otherwise they will be deleted  
    &nbsp;&nbsp;&nbsp;&nbsp;**r**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Reverse time direction in cyclic accumulation  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*t.rast.accumulate* is designed to perform temporal accumulations of
space time raster datasets. This module expects a space time raster
dataset as input that will be sampled by a given **granularity**. All
maps that have the start time during the actual granule will be
accumulated with the predecessor granule accumulation result using the
raster module [r.series.accumulate](r.series.accumulate.md). The default
granularity is 1 day, but any temporal granularity can be set.

The **start** time and the **end** time of the accumulation process must
be set, eg. **start="2000-03-01" end="2011-01-01"**. In addition, a
**cycle**, eg. **cycle="8 months"**, can be specified, that defines
after which interval of time the accumulation process restarts. The
**offset** option specifies the time that should be skipped between two
cycles, eg. **offset="4 months"**.

The **lower** and **upper** **limits** of the accumulation process can
be set, either by using space time raster datasets or by using fixed
values for all raster cells and time steps. The raster maps that specify
the lower and upper limits of the actual granule will be detected using
the following temporal relations: equals, during, overlaps, overlapped
and contains. First, all maps with time stamps equal to the current
granule will be detected, the first lower map and the first upper map
found will be used as limit definitions. If no equal maps are found,
then maps with a temporal during relation are detected, then maps that
temporally overlap the actual granules, until maps that have a temporal
contain relation are detected. If no maps are found or lower/upper STRDS
are not defined, then the **limits** option is used, eg.
**limits=10,30**.

The **upper** **limit** is only used in the Biologically Effective
Degree Days calculation.

The options **shift**, **scale** and **method** are passed to
[r.series.accumulate](r.series.accumulate.md). Please refer to the
manual page of [r.series.accumulate](r.series.accumulate.md) for
detailed option description.

The **output** is a new space time raster dataset with the provided
start time, end time and granularity containing the accumulated raster
maps. The **base** name of the generated maps must always be set. The
**output** space time raster dataset can then be analyzed using
[t.rast.accdetect](t.rast.accdetect.md) to detect specific accumulation
patterns.

## EXAMPLE

This is an example how to accumulate the daily mean temperature of
Europe from 1990 to 2000 using the growing-degree-day method to detect
grass hopper reproduction cycles that are critical to agriculture.

```sh
# Get the temperature data
wget http://www-pool.math.tu-berlin.de/~soeren/grass/temperature_mean_1990_2000_daily_celsius.tar.gz

# Create a temporary project directory
mkdir -p /tmp/grassdata/LL

# Start GRASS and create a new project with PERMANENT mapset
grass -c EPSG:4326 /tmp/grassdata/LL/PERMANENT

# Import the temperature data
t.rast.import input=temperature_mean_1990_2000_daily_celsius.tar.gz \
    output=temperature_mean_1990_2000_daily_celsius directory=/tmp

# We need to set the region correctly
g.region -p raster=`t.rast.list input=temperature_mean_1990_2000_daily_celsius column=name | tail -1`
# We can zoom to the raster map
g.region -p zoom=`t.rast.list input=temperature_mean_1990_2000_daily_celsius column=name | tail -1`

#############################################################################
#### ACCUMULATION USING GDD METHOD ##########################################
#############################################################################
# The computation of grashopper pest control cycles is based on:
#
#   Using Growing Degree Days For Insect Management
#   Nancy E. Adams
#   Extension Educator, Agricultural Resources
#
# available here: http://extension.unh.edu/agric/gddays/docs/growch.pdf

# Now we compute the Biologically Effective Degree Days
# from 1990 - 2000 for each year (12 month cycle) with
# a granularity of one day. Base temperature is 10°C, upper limit is 30°C.
# Hence the accumulation starts at 10°C and does not accumulate values above 30°C.
t.rast.accumulate input="temperature_mean_1990_2000_daily_celsius" \
    output="temperature_mean_1990_2000_daily_celsius_accumulated_10_30" \
    limits="10,30" start="1990-01-01" stop="2000-01-01" cycle="12 months" \
    basename="temp_acc_daily_10_30" method="bedd"

#############################################################################
#### ACCUMULATION PATTERN DETECTION #########################################
#############################################################################
# Now we detect the three grasshopper pest control cycles

# First cycle at 325°C - 427°C GDD
t.rast.accdetect input=temperature_mean_1990_2000_daily_celsius_accumulated_10_30@PERMANENT \
    occ=leafhopper_occurrence_c1_1990_2000 start="1990-01-01" stop="2000-01-01" \
    cycle="12 months" range=325,427 basename=lh_c1 indicator=leafhopper_indicator_c1_1990_2000

# Second cycle at 685°C - 813°C GDD
t.rast.accdetect input=temperature_mean_1990_2000_daily_celsius_accumulated_10_30@PERMANENT \
    occ=leafhopper_occurrence_c2_1990_2000 start="1990-01-01" stop="2000-01-01" \
    cycle="12 months" range=685,813 basename=lh_c2 indicator=leafhopper_indicator_c2_1990_2000

# Third cycle at 1047°C - 1179°C GDD
t.rast.accdetect input=temperature_mean_1990_2000_daily_celsius_accumulated_10_30@PERMANENT \
    occ=leafhopper_occurrence_c3_1990_2000 start="1990-01-01" stop="2000-01-01" \
    cycle="12 months" range=1047,1179 basename=lh_c3 indicator=leafhopper_indicator_c3_1990_2000


#############################################################################
#### YEARLY SPATIAL OCCURRENCE COMPUTATION OF ALL CYCLES ####################
#############################################################################

# Extract the areas that have full cycles
t.rast.aggregate input=leafhopper_indicator_c1_1990_2000 gran="1 year" \
    output=leafhopper_cycle_1_1990_2000_yearly method=maximum basename=li_c1

t.rast.mapcalc input=leafhopper_cycle_1_1990_2000_yearly basename=lh_clean_c1 \
    output=leafhopper_cycle_1_1990_2000_yearly_clean \
    expression="if(leafhopper_cycle_1_1990_2000_yearly == 3, 1, null())"

t.rast.aggregate input=leafhopper_indicator_c2_1990_2000 gran="1 year" \
    output=leafhopper_cycle_2_1990_2000_yearly method=maximum basename=li_c2

t.rast.mapcalc input=leafhopper_cycle_2_1990_2000_yearly basename=lh_clean_c2 \
    output=leafhopper_cycle_2_1990_2000_yearly_clean \
    expression="if(leafhopper_cycle_2_1990_2000_yearly == 3, 2, null())"

t.rast.aggregate input=leafhopper_indicator_c3_1990_2000 gran="1 year" \
    output=leafhopper_cycle_3_1990_2000_yearly method=maximum basename=li_c3

t.rast.mapcalc input=leafhopper_cycle_3_1990_2000_yearly basename=lh_clean_c3 \
    output=leafhopper_cycle_3_1990_2000_yearly_clean \
    expression="if(leafhopper_cycle_3_1990_2000_yearly == 3, 3, null())"


t.rast.mapcalc input=leafhopper_cycle_1_1990_2000_yearly_clean,leafhopper_cycle_2_1990_2000_yearly_clean,leafhopper_cycle_3_1990_2000_yearly_clean \
    basename=lh_cleann_all_cycles \
    output=leafhopper_all_cycles_1990_2000_yearly_clean \
    expression="if(isnull(leafhopper_cycle_3_1990_2000_yearly_clean), \
      if(isnull(leafhopper_cycle_2_1990_2000_yearly_clean), \
      if(isnull(leafhopper_cycle_1_1990_2000_yearly_clean), \
      null() ,1),2),3)"

cat > color.table << EOF
3 yellow
2 blue
1 red
EOF

t.rast.colors input=leafhopper_cycle_1_1990_2000_yearly_clean rules=color.table
t.rast.colors input=leafhopper_cycle_2_1990_2000_yearly_clean rules=color.table
t.rast.colors input=leafhopper_cycle_3_1990_2000_yearly_clean rules=color.table
t.rast.colors input=leafhopper_all_cycles_1990_2000_yearly_clean rules=color.table

#############################################################################
################ DURATION COMPUTATION #######################################
#############################################################################

# Extract the duration in days of the first cycle
t.rast.aggregate input=leafhopper_occurrence_c1_1990_2000 gran="1 year" \
    output=leafhopper_min_day_c1_1990_2000 method=minimum basename=occ_min_day_c1
t.rast.aggregate input=leafhopper_occurrence_c1_1990_2000 gran="1 year" \
    output=leafhopper_max_day_c1_1990_2000 method=maximum basename=occ_max_day_c1
t.rast.mapcalc input=leafhopper_min_day_c1_1990_2000,leafhopper_max_day_c1_1990_2000 \
    basename=occ_duration_c1 \
    output=leafhopper_duration_c1_1990_2000 \
    expression="leafhopper_max_day_c1_1990_2000 - leafhopper_min_day_c1_1990_2000"


# Extract the duration in days of the second cycle
t.rast.aggregate input=leafhopper_occurrence_c2_1990_2000 gran="1 year" \
    output=leafhopper_min_day_c2_1990_2000 method=minimum basename=occ_min_day_c2
t.rast.aggregate input=leafhopper_occurrence_c2_1990_2000 gran="1 year" \
    output=leafhopper_max_day_c2_1990_2000 method=maximum basename=occ_max_day_c2
t.rast.mapcalc input=leafhopper_min_day_c2_1990_2000,leafhopper_max_day_c2_1990_2000 \
    basename=occ_duration_c2 \
    output=leafhopper_duration_c2_1990_2000 \
    expression="leafhopper_max_day_c2_1990_2000 - leafhopper_min_day_c2_1990_2000"


# Extract the duration in days of the third cycle
t.rast.aggregate input=leafhopper_occurrence_c3_1990_2000 gran="1 year" \
    output=leafhopper_min_day_c3_1990_2000 method=minimum basename=occ_min_day_c3
t.rast.aggregate input=leafhopper_occurrence_c3_1990_2000 gran="1 year" \
    output=leafhopper_max_day_c3_1990_2000 method=maximum basename=occ_max_day_c3
t.rast.mapcalc input=leafhopper_min_day_c3_1990_2000,leafhopper_max_day_c3_1990_2000 \
    basename=occ_duration_c3 \
    output=leafhopper_duration_c3_1990_2000 \
    expression="leafhopper_max_day_c3_1990_2000 - leafhopper_min_day_c3_1990_2000"

t.rast.colors input=leafhopper_duration_c1_1990_2000 color=rainbow
t.rast.colors input=leafhopper_duration_c2_1990_2000 color=rainbow
t.rast.colors input=leafhopper_duration_c3_1990_2000 color=rainbow

#############################################################################
################ MONTHLY CYCLES OCCURRENCE ##################################
#############################################################################

# Extract the monthly indicator that shows the start and end of a cycle

# First cycle
t.rast.aggregate input=leafhopper_indicator_c1_1990_2000 gran="1 month" \
    output=leafhopper_indi_min_month_c1_1990_2000 method=minimum basename=occ_indi_min_month_c1
t.rast.aggregate input=leafhopper_indicator_c1_1990_2000 gran="1 month" \
    output=leafhopper_indi_max_month_c1_1990_2000 method=maximum basename=occ_indi_max_month_c1
t.rast.mapcalc input=leafhopper_indi_min_month_c1_1990_2000,leafhopper_indi_max_month_c1_1990_2000 \
    basename=indicator_monthly_c1 \
    output=leafhopper_monthly_indicator_c1_1990_2000 \
    expression="if(leafhopper_indi_min_month_c1_1990_2000 == 1, 1, if(leafhopper_indi_max_month_c1_1990_2000 == 3, 3, 2))"

# Second cycle
t.rast.aggregate input=leafhopper_indicator_c2_1990_2000 gran="1 month" \
    output=leafhopper_indi_min_month_c2_1990_2000 method=minimum basename=occ_indi_min_month_c2
t.rast.aggregate input=leafhopper_indicator_c2_1990_2000 gran="1 month" \
    output=leafhopper_indi_max_month_c2_1990_2000 method=maximum basename=occ_indi_max_month_c2
t.rast.mapcalc input=leafhopper_indi_min_month_c2_1990_2000,leafhopper_indi_max_month_c2_1990_2000 \
    basename=indicator_monthly_c2 \
    output=leafhopper_monthly_indicator_c2_1990_2000 \
    expression="if(leafhopper_indi_min_month_c2_1990_2000 == 1, 1, if(leafhopper_indi_max_month_c2_1990_2000 == 3, 3, 2))"

# Third cycle
t.rast.aggregate input=leafhopper_indicator_c3_1990_2000 gran="1 month" \
    output=leafhopper_indi_min_month_c3_1990_2000 method=minimum basename=occ_indi_min_month_c3
t.rast.aggregate input=leafhopper_indicator_c3_1990_2000 gran="1 month" \
    output=leafhopper_indi_max_month_c3_1990_2000 method=maximum basename=occ_indi_max_month_c3
t.rast.mapcalc input=leafhopper_indi_min_month_c3_1990_2000,leafhopper_indi_max_month_c3_1990_2000 \
    basename=indicator_monthly_c3 \
    output=leafhopper_monthly_indicator_c3_1990_2000 \
    expression="if(leafhopper_indi_min_month_c3_1990_2000 == 1, 1, if(leafhopper_indi_max_month_c3_1990_2000 == 3, 3, 2))"

cat > color.table << EOF
3 red
2 yellow
1 green
EOF

t.rast.colors input=leafhopper_monthly_indicator_c1_1990_2000 rules=color.table
t.rast.colors input=leafhopper_monthly_indicator_c2_1990_2000 rules=color.table
t.rast.colors input=leafhopper_monthly_indicator_c3_1990_2000 rules=color.table

#############################################################################
################ VISUALIZATION ##############################################
#############################################################################
# Now we use g.gui.animation to visualize the yearly occurrence, the duration and the monthly occurrence

# Yearly occurrence of all reproduction cycles
g.gui.animation strds=leafhopper_all_cycles_1990_2000_yearly_clean

# Yearly duration of reproduction cycle 1
g.gui.animation strds=leafhopper_duration_c1_1990_2000
# Yearly duration of reproduction cycle 2
g.gui.animation strds=leafhopper_duration_c2_1990_2000
# Yearly duration of reproduction cycle 3
g.gui.animation strds=leafhopper_duration_c3_1990_2000

# Monthly occurrence of reproduction cycle 1
g.gui.animation strds=leafhopper_monthly_indicator_c1_1990_2000
# Monthly occurrence of reproduction cycle 2
g.gui.animation strds=leafhopper_monthly_indicator_c2_1990_2000
# Monthly occurrence of reproduction cycle 3
g.gui.animation strds=leafhopper_monthly_indicator_c3_1990_2000
```

## REFERENCES

- Jones, G.V., Duff, A.A., Hall, A., Myers, J.W., 2010. Spatial Analysis
  of Climate in Winegrape Growing Regions in the Western United States.
  Am. J. Enol. Vitic. 61, 313-326.

## SEE ALSO

*[t.rast.accdetect](t.rast.accdetect.md),
[t.rast.aggregate](t.rast.aggregate.md),
[t.rast.mapcalc](t.rast.mapcalc.md), [t.info](t.info.md),
[g.region](g.region.md), [r.series.accumulate](r.series.accumulate.md)*

## AUTHOR

Sören Gebbert, Thünen Institute of Climate-Smart Agriculture

## SOURCE CODE

Available at: [t.rast.accumulate source code](https://github.com/OSGeo/grass/tree/main/temporal/t.rast.accumulate)
([history](https://github.com/OSGeo/grass/commits/main/temporal/t.rast.accumulate))  
Latest change: Tuesday Jan 27 12:40:13 2026 in commit [5e8b13d](https://github.com/OSGeo/grass/commit/5e8b13d47dd4fc81efa5abefe90c34f7a8291fb9)
