---
name: r.out.mat
description: Exports a GRASS raster to a binary MAT-File.
keywords: [ raster, export, output ]
---

# r.out.mat

Exports a GRASS raster to a binary MAT-File.

=== "Command line"

    **r.out.mat**
    **input**=*name*
    **output**=*name*
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.out.mat input=name output=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.out.mat***",
        **input**,
        **output**,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.out.mat", input="name", output="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_out_mat*(**input**,
        **output**,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_out_mat(input="name", output="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output binary MAT file  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output binary MAT file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output binary MAT file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.out.mat* will export a GRASS raster map to a MAT-File which can be
loaded into Matlab or Octave for plotting or further analysis.
Attributes such as map title and bounds will also be exported into
additional array variables.  
  
Specifically, the following array variables are created:  

- **map_data**
- **map_name**
- **map_title** (if it exists)
- **map_northern_edge**
- **map_southern_edge**
- **map_eastern_edge**
- **map_western_edge**

In addition, *r.out.mat* makes for a nice binary container format for
transferring georeferenced maps around, even if you don't use Matlab or
Octave.

## NOTES

*r.out.mat* exports a Version 4 MAT-File. These files should
successfully load into more modern versions of Matlab and Octave without
any problems.  
  
Everything should be Endian safe, so the resultant file can be simply
copied between different system architectures without binary
translation.  
  
As there is no IEEE value for `NaN` for integer maps, GRASS's null value
is used to represent it within these maps. You'll have to do something
like this to clean them once the map is loaded into Matlab:

```sh
    map_data(find(map_data < -1e9)) = NaN;
```

Null values in maps containing either floating point or double-precision
floating point data should translate into `NaN` values as expected.  
  
*r.out.mat* must load the entire map into memory before writing,
therefore it might have problems with *huge* maps. (a 3000x4000 DCELL
map uses about 100mb RAM)  
  
GRASS defines its map bounds at the outer-edge of the bounding cells,
not at the coordinates of their centroids. Thus, the following Matlab
commands may be used to determine the map's resolution information:

```sh
    [rows cols] = size(map_data)
    x_range = map_eastern_edge - map_western_edge
    y_range = map_northern_edge - map_southern_edge
    ns_res = y_range/rows
    ew_res = x_range/cols
```

## EXAMPLE

In Matlab, plot with either:

```sh
imagesc(map_data), axis equal, axis tight, colorbar
```

or

```sh
contourf(map_data, 24), axis ij, axis equal, axis tight, colorbar
```

## TODO

Add support for exporting map history, category information, color map,
etc.  
Option to export as a version 5 MAT-File, with map and support
information stored in a single structured array.

## SEE ALSO

*[r.in.mat](r.in.mat.md)  
[r.out.ascii](r.out.ascii.md), [r.out.bin](r.out.bin.md)  
[r.null](r.null.md)  
The [Octave](http://www.octave.org) project*

## AUTHOR

Hamish Bowman  
*Department of Marine Science  
University of Otago  
New Zealand*

## SOURCE CODE

Available at: [r.out.mat source code](https://github.com/OSGeo/grass/tree/main/raster/r.out.mat)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.out.mat))  
Latest change: Thursday May 08 20:58:05 2025 in commit [4144ba8](https://github.com/OSGeo/grass/commit/4144ba81fa125e1549fc52e1cf5feb4fa004be8b)
