---
name: g.message
description: Prints a message, warning, progress info, or fatal error in the GRASS way. This module should be used in scripts for messages served to user.
keywords: [ general, support, scripts ]
---

# g.message

Prints a message, warning, progress info, or fatal error in the GRASS way.

This module should be used in scripts for messages served to user.

=== "Command line"

    **g.message**
    [**-wedpiv**]
    **message**=*string*
    [**debug**=*integer*]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    g.message message=string
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***g.message***",
        **message**,
        **debug**=*1*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("g.message", message="string")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.g_message*(**message**,
        **debug**=*1*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.g_message(message="string")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **message**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Text of the message to be printed  
    &nbsp;&nbsp;&nbsp;&nbsp;Message is printed on GRASS_VERBOSE&gt;=2  
    **debug**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Level to use for debug messages  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-5*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **-w**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print message as warning  
    **-e**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print message as fatal error  
    **-d**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print message as debug message  
    **-p**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print message as progress info  
    **-i**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print message in all modes except of quiet mode  
    &nbsp;&nbsp;&nbsp;&nbsp;Message is printed on GRASS_VERBOSE&gt;=1  
    **-v**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print message only in verbose mode  
    &nbsp;&nbsp;&nbsp;&nbsp;Message is printed only on GRASS_VERBOSE&gt;=3  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **message** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Text of the message to be printed  
    &nbsp;&nbsp;&nbsp;&nbsp;Message is printed on GRASS_VERBOSE&gt;=2  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *string*  
    **debug** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Level to use for debug messages  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-5*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *w*, *e*, *d*, *p*, *i*, *v*  
    &nbsp;&nbsp;&nbsp;&nbsp;**w**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print message as warning  
    &nbsp;&nbsp;&nbsp;&nbsp;**e**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print message as fatal error  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print message as debug message  
    &nbsp;&nbsp;&nbsp;&nbsp;**p**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print message as progress info  
    &nbsp;&nbsp;&nbsp;&nbsp;**i**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print message in all modes except of quiet mode  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Message is printed on GRASS_VERBOSE&gt;=1  
    &nbsp;&nbsp;&nbsp;&nbsp;**v**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print message only in verbose mode  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Message is printed only on GRASS_VERBOSE&gt;=3  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **message** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Text of the message to be printed  
    &nbsp;&nbsp;&nbsp;&nbsp;Message is printed on GRASS_VERBOSE&gt;=2  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *string*  
    **debug** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Level to use for debug messages  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-5*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *w*, *e*, *d*, *p*, *i*, *v*  
    &nbsp;&nbsp;&nbsp;&nbsp;**w**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print message as warning  
    &nbsp;&nbsp;&nbsp;&nbsp;**e**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print message as fatal error  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print message as debug message  
    &nbsp;&nbsp;&nbsp;&nbsp;**p**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print message as progress info  
    &nbsp;&nbsp;&nbsp;&nbsp;**i**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print message in all modes except of quiet mode  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Message is printed on GRASS_VERBOSE&gt;=1  
    &nbsp;&nbsp;&nbsp;&nbsp;**v**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print message only in verbose mode  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Message is printed only on GRASS_VERBOSE&gt;=3  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*g.message* prints a message, warning, progress info, or fatal error in
the GRASS way. This program is to be used in Shell/Perl/Python
scripts, so the author does not need to use the `echo` program. The
advantage of *g.message* is that it formats messages just like other
GRASS modules do and that its functionality is influenced by the
`GRASS_VERBOSE` and `GRASS_MESSAGE_FORMAT` environment variables.

The program can be used for standard informative messages as well as
warnings (**-w** flag) and fatal errors (**-e** flag). For debugging
purposes, the **-d** flag will cause *g.message* to print a debugging
message at the given level.

## NOTES

Messages containing "`=`" must use the full **message=** syntax so the
parser doesn't get confused.

If you want a long message (multi-line) to be dealt with as a single
paragraph, use a single call to *g.message* with text split in the
script using the backslash as the last character. (In shell scripts
don't close the "quote")

A blank line may be obtained with

```sh
g.message message=""
```

Redundant whitespace will be stripped away.

It's advisable to single quote the messages that are to be printed
literally. It prevents a number of characters (most notably, space and
the dollar sign '`$`') from being treated specifically by the shell.

When it is necessary to include, for example, a variable's value as part
of the message, the double quotes may be used, which do not deprive the
dollar sign of its special variable-expansion powers.

While it is known that the interactive Bash instances may treat the
exclamation mark '`!`' character specifically (making single quoting of
it necessary), it shouldn't be the case for the non-interactive
instances of Bash. Nonetheless, to avoid context-based confusion later
on you are encouraged to single-quote messages that do not require
`$VARIABLE` expansion.

### Usage in Python scripts

[GRASS Python Scripting
Library](https://grass.osgeo.org/grass-devel/manuals/libpython/) defines
special wrappers for *g.message*.

- `debug()` for `g.message -d`
- `error()` for `g.message -e`
- `fatal()` for `g.message -e` + `exit()`
- `info()` for `g.message -i`
- `message()` for `g.message`
- `verbose()` for `g.message -v`
- `warning()` for `g.message -w`

Note: The Python tab in the *wxGUI* can be used for entering the
following sample code:

```sh
import grass.script as gcore

gcore.warning("This is a warning")
```

is identical with

```sh
g.message -w message="This is a warning"
```

### VERBOSITY LEVELS

Controlled by the "`GRASS_VERBOSE`" environment variable. Typically this
is set using the **--quiet** or **--verbose** command line options.

- 0 - only errors and warnings are printed
- 1 - progress messages are printed
- 2 - all module messages are printed
- 3 - additional verbose messages are printed

### DEBUG LEVELS

Controlled by the "`DEBUG`" GRASS *gisenv* variable (set with
*[g.gisenv](g.gisenv.md)*).  
Recommended levels:

- 1 - message is printed once or few times per module
- 3 - each row (raster) or line (vector)
- 5 - each cell (raster) or point (vector)

## EXAMPLES

This basic example prints the message "hello" in the console:

```sh
g.message message="hello"
```

To print a message as an error message use the **-e** flag:

```sh
g.message -e message="my error"
```

To print a message highlighted as a debug message ("D0/0: debug") in the
console, use the **-d** flag. Optionally the debug level can be defined
(see also [g.gisenv](g.gisenv.md) for details):

```sh
# Levels: (recommended levels)
#   0 - silence
#   1 - message is printed once or few times per module
#   3 - each row (raster) or line (vector)
#   5 - each cell (raster) or point (vector)
g.message -d message="debug" debug=0
```

To print a message highlighted as a warning message ("WARNING: my
warning") in the console, use the **-w** flag:

```sh
g.message -w message="my warning"
```

## SEE ALSO

*[GRASS variables and environment variables](variables.md)*  
*[g.gisenv](g.gisenv.md), [g.parser](g.parser.md)*

## AUTHOR

Jachym Cepicky

## SOURCE CODE

Available at: [g.message source code](https://github.com/OSGeo/grass/tree/main/general/g.message)
([history](https://github.com/OSGeo/grass/commits/main/general/g.message))  
Latest change: Friday Aug 08 09:39:19 2025 in commit [ed72c71](https://github.com/OSGeo/grass/commit/ed72c71d8580e872cd6fd345c84a15c324764465)
