---
name: g.findfile
description: Searches for GRASS data base files and sets variables for the shell.
keywords: [ general, map management, scripts ]
---

# g.findfile

Searches for GRASS data base files and sets variables for the shell.

=== "Command line"

    **g.findfile**
    [**-nlt**]
    **element**=*string*
    **file**=*string*
    [**mapset**=*string*]
    **format**=*name*
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    g.findfile element=string file=string format=shell
    ```

=== "Python (grass.script)"

    *grass.script.parse_command*("***g.findfile***",
        **element**,
        **file**,
        **mapset**=*None*,
        **format**=*"shell"*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.parse_command("g.findfile", element="string", file="string", format="json")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.g_findfile*(**element**,
        **file**,
        **mapset**=*None*,
        **format**=*"shell"*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.g_findfile(element="string", file="string", format="json")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **element**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of an element  
    **file**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of an existing map  
    **mapset**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of a mapset (default: search path)  
    &nbsp;&nbsp;&nbsp;&nbsp;'.' for current mapset  
    **format**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *shell, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *shell*  
    &nbsp;&nbsp;&nbsp;&nbsp;**shell**: shell script style output  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    **-n**  
    &nbsp;&nbsp;&nbsp;&nbsp;Do not add quotes  
    &nbsp;&nbsp;&nbsp;&nbsp;Ignored when format is set to JSON  
    **-l**  
    &nbsp;&nbsp;&nbsp;&nbsp;List available elements and exit  
    **-t**  
    &nbsp;&nbsp;&nbsp;&nbsp;Return code 0 when file found, 1 otherwise  
    &nbsp;&nbsp;&nbsp;&nbsp;Behave like the test utility, 0 for true, 1 for false, no output  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **element** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of an element  
    **file** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of an existing map  
    **mapset** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of a mapset (default: search path)  
    &nbsp;&nbsp;&nbsp;&nbsp;'.' for current mapset  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *shell, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;**shell**: shell script style output  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *shell*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *n*, *l*, *t*  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Do not add quotes  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Ignored when format is set to JSON  
    &nbsp;&nbsp;&nbsp;&nbsp;**l**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;List available elements and exit  
    &nbsp;&nbsp;&nbsp;&nbsp;**t**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Return code 0 when file found, 1 otherwise  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Behave like the test utility, 0 for true, 1 for false, no output  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **element** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of an element  
    **file** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of an existing map  
    **mapset** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of a mapset (default: search path)  
    &nbsp;&nbsp;&nbsp;&nbsp;'.' for current mapset  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *shell, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;**shell**: shell script style output  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *shell*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *n*, *l*, *t*  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Do not add quotes  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Ignored when format is set to JSON  
    &nbsp;&nbsp;&nbsp;&nbsp;**l**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;List available elements and exit  
    &nbsp;&nbsp;&nbsp;&nbsp;**t**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Return code 0 when file found, 1 otherwise  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Behave like the test utility, 0 for true, 1 for false, no output  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*g.findfile* is designed for Bourne shell or Python scripts that need to
search for mapset *elements*, including: raster, vector maps, region
definitions and *[imagery](i.group.md)* groups.

The list of **element** names to search for is not fixed; any
subdirectory of the mapset directory is a valid **element** name.

However, the user can find the list of standard GRASS **element** names
in the file `$GISBASE/etc/element_list`. This is the file which
*[g.remove](g.remove.md)*, *[g.rename](g.rename.md)* and
*[g.copy](g.copy.md)* use to determine which files need to be
deleted/renamed/copied for a given entity type.

## NOTES

*g.findfile* writes four lines to standard output:

```sh
name='file_name'
mapset='mapset_name'
file='unix_filename'
fullname='grass_fullname'
```

The output is *Bash* commands to set the variable *name* to the GRASS
data base file name, *mapset* to the mapset in which the file resides,
and *file* to the full UNIX path name for the named file. These
variables may be set in the *Bash* as follows:

```sh
eval `g.findfile element=name mapset=name file=name`
```

## EXAMPLES

### SHELL

**Raster map example:**

```sh
eval `g.findfile element=cell file=elevation`
```

If the specified file (here: raster map) does not exist, the variables
will be set as follows:

```sh
name=
mapset=
fullname=
file=
```

The following is a way to test for this case:

```sh
if [ ! "$file" ]
then
    exit 1
fi
```

**Vector map example (including error message):**

```sh
eval `g.findfile element=vector file="$G_OPT_V_INPUT"`
if [ ! "$file" ] ; then
   g.message -e "Vector map <$G_OPT_V_INPUT> not found"
   exit 1
fi
```

### PYTHON

See *[Python Scripting
Library](https://grass.osgeo.org/grass-devel/manuals/libpython/)* for
more info.

Note: The Python tab in the *wxGUI* can be used for entering the
following code:

```python
import grass.script as gs

gs.find_file('elevation', element = 'cell')
```

Parsing JSON output:

```python
import grass.script as gs

data = gs.parse_command("g.findfile", file="elevation", element="raster", format="json")
print(data)
```

Possible output:

```text
{'name': 'elevation', 'mapset': 'PERMANENT', 'fullname': 'elevation@PERMANENT', 'file': '/grassdata/PERMANENT/cell/elevation'}
```

## SEE ALSO

*[g.filename](g.filename.md), [g.gisenv](g.gisenv.md),
[g.mapsets](g.mapsets.md), [g.parser](g.parser.md)*

## AUTHOR

Michael Shapiro, U.S.Army Construction Engineering Research Laboratory

## SOURCE CODE

Available at: [g.findfile source code](https://github.com/OSGeo/grass/tree/main/general/g.findfile)
([history](https://github.com/OSGeo/grass/commits/main/general/g.findfile))  
Latest change: Monday Nov 03 07:16:48 2025 in commit [c967967](https://github.com/OSGeo/grass/commit/c96796774942e46bbababcd652d3986d4e295665)
