---
name: r.walk
description: Creates a raster map showing the anisotropic cumulative cost of moving between different geographic locations on an input raster map whose cell category values represent cost.
keywords: [ raster, cost surface, cumulative costs, cost allocation ]
---

# r.walk

Creates a raster map showing the anisotropic cumulative cost of moving between different geographic locations on an input raster map whose cell category values represent cost.

=== "Command line"

    **r.walk**
    [**-knrib**]
    **elevation**=*name*
    **friction**=*name*
    **output**=*name*
    [**solver**=*name*]
    [**nearest**=*name*]
    [**outdir**=*name*]
    [**start_points**=*name*]
    [**stop_points**=*name*]
    [**start_raster**=*name*]
    [**start_coordinates**=*east,north* [,*east,north*,...]]
    [**stop_coordinates**=*east,north* [,*east,north*,...]]
    [**max_cost**=*value*]
    [**null_cost**=*value*]
    [**memory**=*memory in MB*]
    [**walk_coeff**=*a,b,c,d*]
    [**lambda**=*float*]
    [**slope_factor**=*float*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.walk elevation=name friction=name output=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.walk***",
        **elevation**,
        **friction**,
        **output**,
        **solver**=*None*,
        **nearest**=*None*,
        **outdir**=*None*,
        **start_points**=*None*,
        **stop_points**=*None*,
        **start_raster**=*None*,
        **start_coordinates**=*None*,
        **stop_coordinates**=*None*,
        **max_cost**=*0*,
        **null_cost**=*None*,
        **memory**=*300*,
        **walk_coeff**=*"0.72,6.0,1.9998,-1.9998"*,
        **lambda**=*1.0*,
        **slope_factor**=*-0.2125*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.walk", elevation="name", friction="name", output="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_walk*(**elevation**,
        **friction**,
        **output**,
        **solver**=*None*,
        **nearest**=*None*,
        **outdir**=*None*,
        **start_points**=*None*,
        **stop_points**=*None*,
        **start_raster**=*None*,
        **start_coordinates**=*None*,
        **stop_coordinates**=*None*,
        **max_cost**=*0*,
        **null_cost**=*None*,
        **memory**=*300*,
        **walk_coeff**=*"0.72,6.0,1.9998,-1.9998"*,
        **lambda**=*1.0*,
        **slope_factor**=*-0.2125*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_walk(elevation="name", friction="name", output="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **elevation**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map  
    **friction**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map containing friction costs  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map to contain walking costs  
    **solver**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map solving equal costs  
    &nbsp;&nbsp;&nbsp;&nbsp;Helper variable to pick a direction if two directions have equal cumulative costs (smaller is better)  
    **nearest**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map with nearest start point  
    **outdir**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map to contain movement directions  
    **start_points**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of starting vector points map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    **stop_points**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of stopping vector points map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    **start_raster**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of starting raster points map  
    **start_coordinates**=*east,north* [,*east,north*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Coordinates of starting point(s) (E,N)  
    **stop_coordinates**=*east,north* [,*east,north*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Coordinates of stopping point(s) (E,N)  
    **max_cost**=*value*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum cumulative cost  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **null_cost**=*value*  
    &nbsp;&nbsp;&nbsp;&nbsp;Cost assigned to null cells. By default, null cells are excluded  
    **memory**=*memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **walk_coeff**=*a,b,c,d*  
    &nbsp;&nbsp;&nbsp;&nbsp;Coefficients for walking energy formula parameters a,b,c,d  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.72,6.0,1.9998,-1.9998*  
    **lambda**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Lambda coefficients for combining walking energy and friction cost  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.0*  
    **slope_factor**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Slope factor determines travel energy cost per height step  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-0.2125*  
    **-k**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use the 'Knight's move'; slower, but more accurate  
    **-n**  
    &nbsp;&nbsp;&nbsp;&nbsp;Keep null values in output raster map  
    **-r**  
    &nbsp;&nbsp;&nbsp;&nbsp;Start with values in raster map  
    **-i**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print info about disk space and memory requirements and exit  
    **-b**  
    &nbsp;&nbsp;&nbsp;&nbsp;Create bitmask encoded directions  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **elevation** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **friction** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map containing friction costs  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map to contain walking costs  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **solver** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map solving equal costs  
    &nbsp;&nbsp;&nbsp;&nbsp;Helper variable to pick a direction if two directions have equal cumulative costs (smaller is better)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **nearest** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map with nearest start point  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **outdir** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map to contain movement directions  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **start_points** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of starting vector points map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **stop_points** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of stopping vector points map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **start_raster** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of starting raster points map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **start_coordinates** : list[tuple[float, float]] | tuple[float, float] | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Coordinates of starting point(s) (E,N)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, coords, *east,north*  
    **stop_coordinates** : list[tuple[float, float]] | tuple[float, float] | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Coordinates of stopping point(s) (E,N)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, coords, *east,north*  
    **max_cost** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum cumulative cost  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *value*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **null_cost** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Cost assigned to null cells. By default, null cells are excluded  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *value*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **walk_coeff** : tuple[str, str, str, str] | list[str] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Coefficients for walking energy formula parameters a,b,c,d  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *a,b,c,d*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.72,6.0,1.9998,-1.9998*  
    **lambda** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Lambda coefficients for combining walking energy and friction cost  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.0*  
    **slope_factor** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Slope factor determines travel energy cost per height step  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-0.2125*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *k*, *n*, *r*, *i*, *b*  
    &nbsp;&nbsp;&nbsp;&nbsp;**k**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use the 'Knight's move'; slower, but more accurate  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Keep null values in output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;**r**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Start with values in raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;**i**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print info about disk space and memory requirements and exit  
    &nbsp;&nbsp;&nbsp;&nbsp;**b**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Create bitmask encoded directions  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **elevation** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **friction** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map containing friction costs  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map to contain walking costs  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **solver** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map solving equal costs  
    &nbsp;&nbsp;&nbsp;&nbsp;Helper variable to pick a direction if two directions have equal cumulative costs (smaller is better)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **nearest** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map with nearest start point  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **outdir** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map to contain movement directions  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **start_points** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of starting vector points map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **stop_points** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of stopping vector points map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **start_raster** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of starting raster points map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **start_coordinates** : list[tuple[float, float]] | tuple[float, float] | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Coordinates of starting point(s) (E,N)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, coords, *east,north*  
    **stop_coordinates** : list[tuple[float, float]] | tuple[float, float] | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Coordinates of stopping point(s) (E,N)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, coords, *east,north*  
    **max_cost** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum cumulative cost  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *value*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **null_cost** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Cost assigned to null cells. By default, null cells are excluded  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *value*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **walk_coeff** : tuple[str, str, str, str] | list[str] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Coefficients for walking energy formula parameters a,b,c,d  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *a,b,c,d*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.72,6.0,1.9998,-1.9998*  
    **lambda** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Lambda coefficients for combining walking energy and friction cost  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.0*  
    **slope_factor** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Slope factor determines travel energy cost per height step  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-0.2125*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *k*, *n*, *r*, *i*, *b*  
    &nbsp;&nbsp;&nbsp;&nbsp;**k**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use the 'Knight's move'; slower, but more accurate  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Keep null values in output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;**r**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Start with values in raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;**i**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print info about disk space and memory requirements and exit  
    &nbsp;&nbsp;&nbsp;&nbsp;**b**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Create bitmask encoded directions  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.walk* computes anisotropic cumulative cost of moving between
different geographic locations on an input elevation raster map whose
cell category values represent elevation combined with an input raster
map layer whose cell values represent friction cost.

*r.walk* outputs 1) a raster map showing the lowest cumulative cost
(time) of moving between each cell and the user-specified starting
points and 2) a second raster map showing the movement direction to the
next cell on the path back to the start point (see [Movement
Direction](#movement-direction)). It uses an input elevation raster map whose cell
category values represent elevation, combined with a second input raster
map whose cell values represent friction costs.

This function is similar to *[r.cost](r.cost.md)*, but in addition to a
friction map, it considers an anisotropic travel time due to the
different walking speed associated with downhill and uphill movements.

## NOTES

The formula from Aitken 1977/Langmuir 1984 (based on Naismith's rule for
walking times) has been used to estimate the cost parameters of specific
slope intervals:

```sh
T = a*delta_S + b*delta_H_uphill + c*delta_H_moderate_downhill + d*delta_H_steep_downhill
```

where:

- `T` is time of movement in seconds,
- `delta S` is the horizontal distance covered in meters,
- `delta H` is the altitude difference in meters.

The a, b, c, d **walk_coeff** parameters take in account movement speed
in the different conditions and are linked to:

- a: time in seconds it takes to walk for 1 meter a flat surface
  (1/walking speed)
- b: additional walking time in seconds, per meter of elevation gain on
  uphill slopes
- c: additional walking time in seconds, per meter of elevation loss on
  moderate downhill slopes (use positive value for decreasing cost)
- d: additional walking time in seconds, per meter of elevation loss on
  steep downhill slopes (use negative value for increasing cost)

It has been proved that moving downhill is favourable up to a specific
slope value threshold, after that it becomes unfavourable. The default
slope value threshold (**slope_factor**) is -0.2125, corresponding to
tan(-12), calibrated on human behaviour (\>5 and \<12 degrees: moderate
downhill; \>12 degrees: steep downhill). The default values for a, b, c,
d **walk_coeff** parameters are those proposed by Langmuir (0.72, 6.0,
1.9998, -1.9998), based on man walking effort in standard conditions.

The **friction** cost parameter represents a time penalty in seconds of
additional walking time to cross 1 meter distance. Friction cost can be
any floating point value ≥ 0. A friction map is a required parameter; if
no friction costs are desired, a friction map should be a raster in
which all cells have a value of 0.

The **lambda** parameter is a dimensionless scaling factor of the
friction cost:

```sh
total cost = movement time cost + lambda * friction costs * delta_S
```

For a more accurate result, the "knight's move" option can be used
(although it is more time consuming). In the diagram below, the center
location (O) represents a grid cell from which cumulative distances are
calculated. Those neighbours marked with an x are always considered for
cumulative cost updates. With the "knight's move" option, the neighbours
marked with a K are also considered.

```sh
  K   K
K x x x K
  x O x
K x x x K
  K   K
```

The minimum cumulative costs are computed using Dijkstra's algorithm,
that find an optimum solution (for more details see *r.cost*, that uses
the same algorithm).

### Movement Direction

The movement direction surface is created to record the sequence of
movements that created the cost accumulation surface. This movement
direction surface can be used by *[r.path](r.path.md)* to recover a path
from an end point back to the start point. The direction of each cell
points towards the next cell. The directions are recorded as degrees CCW
from East:

```sh
       112.5      67.5         i.e. a cell with the value 135
157.5  135   90   45   22.5    means the next cell is to the north-west
       180   x   360
202.5  225  270  315  337.5
       247.5     292.5
```

Once *r.walk* computes the cumulative cost map as a linear combination
of friction cost (from friction map) and the altitude and distance
covered (from the digital elevation model), the associated movement
direction map can be used by *[r.path](r.path.md)* to find the minimum
cost path.

*r.walk*, like most all GRASS raster programs, is also made to be run on
maps larger that can fit in available computer memory. As the algorithm
works through the dynamic list of cells it can move almost randomly
around the entire area. *r.walk* divides the entire area into a number
of pieces and swaps these pieces in and out of memory (to and from disk)
as needed. This provides a virtual memory approach optimally designed
for 2-D raster maps. The amount of memory to be used by *r.walk* can be
controlled with the **memory** option, default is 300 MB. For systems
with less memory this value will have to be set to a lower value.

## EXAMPLES

We compute a map showing how far a lost person could get from the point
where he or she was last seen while taking into account the topography
and landcover.

```sh
g.region swwake_30m -p

# create friction map based on land cover
r.recode landclass96 out=friction rules=- << EOF
1:3:0.1:0.1
4:5:10.:10.
6:6:1000.0:1000.0
7:7:0.3:0.3
EOF

r.walk -k elevation=elev_ned_30m friction=friction output=walkcost \
    start_coordinates=635576,216485 lambda=0.5 max=10000

# compute contours on the cost surface to better understand
# how far the person can get in certain time (1000 is in seconds)
r.contour walkcost output=walkcost step=1000
```

![r.walk example](r_walk.png)  
*Figure: Walkshed over a cost surface derived from topography and
landcover*

## REFERENCES

- Aitken, R. 1977. Wilderness areas in Scotland. Unpublished Ph.D.
  thesis. University of Aberdeen.
- Steno Fontanari, University of Trento, Italy, Ingegneria per
  l'Ambiente e il Territorio, 2000-2001.
- Langmuir, E. 1984. Mountaincraft and leadership. The Scottish Sports
  Council/MLTB. Cordee, Leicester.

## SEE ALSO

*[r.cost](r.cost.md), [r.path](r.path.md), [r.in.ascii](r.in.ascii.md),
[r.mapcalc](r.mapcalc.md), [r.recode](r.recode.md),
[r.out.ascii](r.out.ascii.md)*

## AUTHORS

**Based on r.cost written by :**  
Antony Awaida, Intelligent Engineering, Systems Laboratory, M.I.T.  
James Westervelt, U.S.Army Construction Engineering Research
Laboratory  
Updated for Grass 5 by Pierre de Mouveaux (<pmx@audiovu.com>)

**Initial version of r.walk:**  
Steno Fontanari, 2002

**Current version of r.walk:**  
Franceschetti Simone, Sorrentino Diego, Mussi Fabiano and Pasolli
Mattia  
Correction by: Fontanari Steno, Napolitano Maurizio and Flor Roberto  
In collaboration with: Franchi Matteo, Vaglia Beatrice, Bartucca Luisa,
Fava Valentina and Tolotti Mathias, 2004

**Updated for GRASS 6.1:**  
Roberto Flor and Markus Neteler

**Updated for GRASS GIS 7:**  
Markus Metz  
Multiple path directions sponsored by
[mundialis](https://www.mundialis.de)

## SOURCE CODE

Available at: [r.walk source code](https://github.com/OSGeo/grass/tree/main/raster/r.walk)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.walk))  
Latest change: Friday Mar 07 07:39:48 2025 in commit [e1e37d8](https://github.com/OSGeo/grass/commit/e1e37d87e78212dfcf4c97aaf32c54042adbd2b0)
