---
name: r.buffer
description: Creates a raster map showing buffer zones surrounding cells that contain non-NULL category values.
keywords: [ raster, buffer ]
---

# r.buffer

Creates a raster map showing buffer zones surrounding cells that contain non-NULL category values.

=== "Command line"

    **r.buffer**
    [**-z**]
    **input**=*name*
    **output**=*name*
    **distances**=*float* [,*float*,...]
    [**units**=*string*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.buffer input=name output=name distances=0.0
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.buffer***",
        **input**,
        **output**,
        **distances**,
        **units**=*"meters"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.buffer", input="name", output="name", distances=0.0)
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_buffer*(**input**,
        **output**,
        **distances**,
        **units**=*"meters"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_buffer(input="name", output="name", distances=0.0)
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    **distances**=*float* [,*float*,...] **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Distance zone(s)  
    **units**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Units of distance  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *meters, kilometers, feet, miles, nautmiles*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *meters*  
    **-z**  
    &nbsp;&nbsp;&nbsp;&nbsp;Ignore zero (0) data cells instead of NULL cells  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **distances** : float | list[float] | str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Distance zone(s)  
    **units** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Units of distance  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *meters, kilometers, feet, miles, nautmiles*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *meters*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *z*  
    &nbsp;&nbsp;&nbsp;&nbsp;**z**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Ignore zero (0) data cells instead of NULL cells  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **distances** : float | list[float] | str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Distance zone(s)  
    **units** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Units of distance  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *meters, kilometers, feet, miles, nautmiles*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *meters*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *z*  
    &nbsp;&nbsp;&nbsp;&nbsp;**z**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Ignore zero (0) data cells instead of NULL cells  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.buffer* creates a new raster map showing buffer (a.k.a. "distance" or
"proximity") zones around all cells that contain non-NULL category
values in an existing raster map. The distances of buffer zones from
cells with non-zero category values are user-chosen and must be positive
value(s). Suppose, for example, that you want to place buffer zones
around roads. This program could create the raster map shown below on
the right based on road information contained in the raster map shown on
the left.

```sh
      000000000000000000000000 222233333333333333444444
      111000000000000000000000 111222222222223333333333
      000111111111100000000000 222111111111122223333333
      000000001000011100000000 332222221222211122222222
      000000001000000011111111 333333321233222211111111
      000000001000000000000000 433333321233333222222222
      000000001000000000000000 444443321233333333333333
      000000001000000000000000 444443321233443333333333
      000000001000000000000000 444443321233444444444444

      Category 0: No roads
      Category 1: Road location
      Category 2: Buffer Zone 1 around roads
      Category 3: Buffer Zone 2 around roads
      Category 4: Buffer Zone 3 around roads
```

## NOTES

The user has the option of identifying up to 250 continuous zones. The
zones are identified by specifying the upper limit of each desired zone
(*r.buffer* assumes that `0` is the starting point). "Continuous" is
used in the sense that each category zone's lower value is the previous
zone's upper value. The first buffer zone always has distance `0` as its
lower bound. Buffer distances can be specified using one of five units
with the **units** parameter.

Distances from cells containing the user-specified category values are
calculated using the "fromcell" method. This method locates each cell
that contains a category value from which distances are to be
calculated, and draws the requested distance rings around them. This
method works very fast when there are few cells containing the category
values of interest, but works slowly when there are numerous cells
containing the category values of interest spread throughout the area.

*r.buffer* measures distances from center of cell to center of cell
using Euclidean distance measure for planimetric coordinate reference
systems (like UTM) and using ellipsoidal geodesic distance measure for
latitude/longitude CRS.

*r.buffer* calculates distance zones from all cells having non-NULL
category values in the **input** map. If the user wishes to calculate
distances from only selected **input** map category values, the user
should run (for example) *[r.reclass](r.reclass.md)* prior to
*r.buffer*, to reclass all categories from which distance zones are not
desired to be calculated into category NULL.

The **-z** flag can be used to ignore raster values of zero instead of
NULL values in the input raster map.

When working with massive raster regions consider the
*[r.buffer.lowmem](r.buffer.lowmem.md)* module if RAM use becomes a
problem. The lowmem version can be \> 40x slower, but will work with
minimal memory requirements. The classic *r.buffer* should be able to
deal with raster maps of 32000x32000 size on a system with 1 GB RAM, and
rasters of 90000x90000 on a system with 8 GB RAM without going into
swap.

## EXAMPLE

In the following example (North Carolina sample dataset), the buffer
zones would be (in the default map units of meters): 0-100, 101-200,
201-300, 301-400 and 401-500.  

```sh
g.region raster=roadsmajor -p
r.buffer input=roadsmajor output=roadsmajor_buf distances=100,200,300,400,500
```

Result:

```sh
r.category input=roads.buf
      1       distances calculated from these locations
      2       0-100 meters
      3       100-200 meters
      4       200-300 meters
      5       300-400 meters
      6       400-500 meters
```

![Distances to road](r_buffer_road.png)  
*Distances to road*

## SEE ALSO

*[r.buffer.lowmem](r.buffer.lowmem.md), [r.grow](r.grow.md),
[v.buffer](v.buffer.md)*

*[g.region](g.region.md), [r.cost](r.cost.md),
[r.distance](r.distance.md), [r.grow.distance](r.grow.distance.md),
[r.mapcalc](r.mapcalc.md), [r.reclass](r.reclass.md)*

## AUTHORS

Michael Shapiro, U.S. Army Construction Engineering Research
Laboratory  
James Westervelt, U.S. Army Construction Engineering Research Laboratory

## SOURCE CODE

Available at: [r.buffer source code](https://github.com/OSGeo/grass/tree/main/raster/r.buffer)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.buffer))  
Latest change: Wednesday Oct 15 09:22:00 2025 in commit [e64e777](https://github.com/OSGeo/grass/commit/e64e7774967984ecd55169f76abdaac35de1ad21)
