---
name: r.surf.fractal
description: Creates a fractal surface of a given fractal dimension.
keywords: [ raster, surface, fractal ]
---

# r.surf.fractal

Creates a fractal surface of a given fractal dimension.

=== "Command line"

    **r.surf.fractal**
    **output**=*name*
    [**dimension**=*float*]
    [**number**=*integer*]
    [**seed**=*integer*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.surf.fractal output=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.surf.fractal***",
        **output**,
        **dimension**=*2.05*,
        **number**=*0*,
        **seed**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.surf.fractal", output="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_surf_fractal*(**output**,
        **dimension**=*2.05*,
        **number**=*0*,
        **seed**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_surf_fractal(output="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    **dimension**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Fractal dimension of surface (2 &lt; D &lt; 3)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2.05*  
    **number**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of intermediate images to produce  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **seed**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Seed value for the random number generator  
    &nbsp;&nbsp;&nbsp;&nbsp;Using the same seed ensures identical results, while a randomly generated seed produces different outcomes in each run.  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **dimension** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Fractal dimension of surface (2 &lt; D &lt; 3)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2.05*  
    **number** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of intermediate images to produce  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **seed** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Seed value for the random number generator  
    &nbsp;&nbsp;&nbsp;&nbsp;Using the same seed ensures identical results, while a randomly generated seed produces different outcomes in each run.  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **dimension** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Fractal dimension of surface (2 &lt; D &lt; 3)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2.05*  
    **number** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of intermediate images to produce  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **seed** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Seed value for the random number generator  
    &nbsp;&nbsp;&nbsp;&nbsp;Using the same seed ensures identical results, while a randomly generated seed produces different outcomes in each run.  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

**r.surf.fractal** creates a fractal surface of a given fractal
dimension. It uses the spectral synthesis method. The module can create
intermediate layers showing the build up of different spectral
coefficients (see Saupe, pp.106-107 for an example of this).

This module generates naturally looking synthetical elevation models
(DEM).

## NOTE

This module requires the [FFTW library](https://fftw.org) for computing
Discrete Fourier Transforms.

## EXAMPLE

Generate surface using fractals in selected region, set color table and
display with shade.

```sh
g.region -p raster=elevation

r.surf.fractal output=fractals

r.colors map=fractals color=byr
r.relief input=fractals output=fractals_shade

d.mon wx0
d.shade shade=fractals_shade color=fractals b=50
```

![Artificial surface created with fractals](r_surf_fractal_simple.png)  
Artificial surface created with fractals  

Compare results when using different fractal dimensions:

```sh
# D=2.0005
g.region -dp
r.surf.fractal out=dem_d2_0005 dim=2.0005
r.info -r dem_d2_0005
r.mapcalc "dem_d2_0005_final = 1.0 * dem_d2_0005 + abs(min(dem_d2_0005))"
r.colors dem_d2_0005_final color=terrain
r.slope.aspect dem_d2_0005_final aspect=dem_d2_0005_final_as

# D=2.90
r.surf.fractal out=dem_d2_90 dim=2.90
r.info -r dem_d2_90
r.mapcalc "dem_d2_90_final = 1.0 * dem_d2_90 + abs(min(dem_d2_90))"
r.colors dem_d2_90_final color=terrain
r.slope.aspect dem_d2_90_final aspect=dem_d2_90_final_as
```

![Artificial DEMs created with fractals](r_surf_fractal.jpg)  
Artificial DEMs created with fractals:  
top: fractal dimension d=2.0005 (left: elevation map, right: aspect
map)  
top: fractal dimension d=2.90 (left: elevation map, right: aspect map)

## REFERENCES

Saupe, D. (1988) Algorithms for random fractals, in Barnsley M., Devaney
R., Mandelbrot B., Peitgen, H-O., Saupe D., and Voss R. (1988) The
Science of Fractal Images, Ch. 2, pp.71-136. London: Springer-Verlag.

## SEE ALSO

*[r.surf.contour](r.surf.contour.md), [r.surf.idw](r.surf.idw.md),
[r.surf.gauss](r.surf.gauss.md), [r.surf.random](r.surf.random.md),
[v.surf.idw](v.surf.idw.md), [v.surf.rst](v.surf.rst.md)*

## AUTHOR

Jo Wood, Midlands Regional Research Laboratory (ASSIST), University of
Leicester

## SOURCE CODE

Available at: [r.surf.fractal source code](https://github.com/OSGeo/grass/tree/main/raster/r.surf.fractal)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.surf.fractal))  
Latest change: Thursday May 08 20:58:05 2025 in commit [4144ba8](https://github.com/OSGeo/grass/commit/4144ba81fa125e1549fc52e1cf5feb4fa004be8b)
