---
name: i.modis.qc
description: Extracts quality control parameters from MODIS QC layers.
keywords: [ imagery, imagery quality assessment, reflectance, land surface temperature, vegetation, satellite, MODIS ]
---

# i.modis.qc

Extracts quality control parameters from MODIS QC layers.

=== "Command line"

    **i.modis.qc**
    **input**=*name*
    **output**=*name*
    **productname**=*string*
    **qcname**=*string*
    [**band**=*string*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    i.modis.qc input=name output=name productname=mod09Q1 qcname=adjcorr
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***i.modis.qc***",
        **input**,
        **output**,
        **productname**,
        **qcname**,
        **band**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("i.modis.qc", input="name", output="name", productname="mod09Q1", qcname="adjcorr")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.i_modis_qc*(**input**,
        **output**,
        **productname**,
        **qcname**,
        **band**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.i_modis_qc(input="name", output="name", productname="mod09Q1", qcname="adjcorr")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input surface reflectance QC layer [bit array]  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output QC type classification layer  
    **productname**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of MODIS product type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *mod09Q1, mod09A1, mod09A1s, mod09GA, mod09GAs, mod09CMG, mod09CMGs, mod09CMGi, mod11A1, mod11A2, mod13A2, mcd43B2, mcd43B2q, mod13Q1*  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09Q1**: surf. refl. 250m 8-days  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09A1**: surf. refl. 500m 8-days  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09A1s**: surf. refl. 500m 8-days, State QA  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09GA**: surf. refl. 500m daily  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09GAs**: surf. refl. 500m daily, State QA  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09CMG**: surf. refl. 5000m daily  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09CMGs**: surf. refl. 5000m daily, State QA  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09CMGi**: surf. refl. 5000m daily, Internal Climatology  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod11A1**: LST 1km daily (Day/Night)  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod11A2**: LST 1km 8-days (Day/Night)  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod13A2**: VI 1km 16-days  
    &nbsp;&nbsp;&nbsp;&nbsp;**mcd43B2**: Brdf-Albedo Quality (Ancillary SDS) 1km 8-days  
    &nbsp;&nbsp;&nbsp;&nbsp;**mcd43B2q**: Brdf-Albedo Quality (BRDF SDS) 1km 8-days  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod13Q1**: VI 250m 16-days  
    **qcname**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of QC type to extract  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *adjcorr, atcorr, cloud, data_quality, diff_orbit_from_500m, modland_qa, mandatory_qa_11A1, data_quality_flag_11A1, emis_error_11A1, lst_error_11A1, data_quality_flag_11A2, emis_error_11A2, mandatory_qa_11A2, lst_error_11A2, aerosol_quantity, brdf_correction_performed, cirrus_detected, cloud_shadow, cloud_state, internal_cloud_algorithm, internal_fire_algorithm, internal_snow_mask, land_water, mod35_snow_ice, pixel_adjacent_to_cloud, salt_pan, icm_cloudy, icm_clear, icm_high_clouds, icm_low_clouds, icm_snow, icm_fire, icm_sun_glint, icm_dust, icm_cloud_shadow, icm_pixel_is_adjacent_to_cloud, icm_cirrus, icm_pan_flag, icm_criteria_for_aerosol_retrieval, icm_aot_has_clim_val, modland_qa, vi_usefulness, aerosol_quantity, pixel_adjacent_to_cloud, brdf_correction_performed, mixed_clouds, land_water, possible_snow_ice, possible_shadow, platform, land_water, sun_z_angle_at_local_noon, brdf_correction_performed, modland_qa, vi_usefulness, aerosol_quantity, pixel_adjacent_to_cloud, brdf_correction_performed, mixed_clouds, land_water, possible_snow_ice, possible_shadow*  
    &nbsp;&nbsp;&nbsp;&nbsp;**adjcorr**: mod09: Adjacency Correction  
    &nbsp;&nbsp;&nbsp;&nbsp;**atcorr**: mod09: Atmospheric Correction  
    &nbsp;&nbsp;&nbsp;&nbsp;**cloud**: mod09: Cloud State  
    &nbsp;&nbsp;&nbsp;&nbsp;**data_quality**: mod09: Band-Wise Data Quality Flag  
    &nbsp;&nbsp;&nbsp;&nbsp;**diff_orbit_from_500m**: mod09: 250m Band is at Different Orbit than 500m  
    &nbsp;&nbsp;&nbsp;&nbsp;**modland_qa**: mod13Q1: MODIS Land General Quality Assessment  
    &nbsp;&nbsp;&nbsp;&nbsp;**mandatory_qa_11A1**: mod11A1: MODIS Land General Quality Assessment  
    &nbsp;&nbsp;&nbsp;&nbsp;**data_quality_flag_11A1**: mod11A1: Detailed Quality Indications  
    &nbsp;&nbsp;&nbsp;&nbsp;**emis_error_11A1**: mod11A1: Average Emissivity Error Classes  
    &nbsp;&nbsp;&nbsp;&nbsp;**lst_error_11A1**: mod11A1: Average LST Error Classes  
    &nbsp;&nbsp;&nbsp;&nbsp;**data_quality_flag_11A2**: mod11A2: Detailed Quality Indications  
    &nbsp;&nbsp;&nbsp;&nbsp;**emis_error_11A2**: mod11A2: Average Emissivity Error Classes  
    &nbsp;&nbsp;&nbsp;&nbsp;**mandatory_qa_11A2**: mod11A2: MODIS Land General Quality Assessment  
    &nbsp;&nbsp;&nbsp;&nbsp;**lst_error_11A2**: mod11A2: Average LST Error Classes  
    &nbsp;&nbsp;&nbsp;&nbsp;**aerosol_quantity**: mod13Q1: Quantity range of Aerosol  
    &nbsp;&nbsp;&nbsp;&nbsp;**brdf_correction_performed**: mod13Q1: if BRDF correction performed  
    &nbsp;&nbsp;&nbsp;&nbsp;**cirrus_detected**: mod09\*s: StateQA Cirrus Detected  
    &nbsp;&nbsp;&nbsp;&nbsp;**cloud_shadow**: mod09\*s: StateQA Cloud Shadow  
    &nbsp;&nbsp;&nbsp;&nbsp;**cloud_state**: mod09\*s: StateQA Cloud State  
    &nbsp;&nbsp;&nbsp;&nbsp;**internal_cloud_algorithm**: mod09\*s: StateQA Internal Cloud Algorithm  
    &nbsp;&nbsp;&nbsp;&nbsp;**internal_fire_algorithm**: mod09\*s: StateQA Internal Fire Algorithm  
    &nbsp;&nbsp;&nbsp;&nbsp;**internal_snow_mask**: mod09\*s: StateQA Internal Snow Mask  
    &nbsp;&nbsp;&nbsp;&nbsp;**land_water**: mod13Q1: separate land from various water objects  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod35_snow_ice**: mod09\*s: StateQA mod35 Snow Ice  
    &nbsp;&nbsp;&nbsp;&nbsp;**pixel_adjacent_to_cloud**: mod13Q1: if pixel is a cloud neighbour  
    &nbsp;&nbsp;&nbsp;&nbsp;**salt_pan**: mod09\*s: StateQA Salt Pan (mod09GAs)  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_cloudy**: mod09\*i: Internal CM: Cloudy  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_clear**: mod09\*i: Internal CM: Clear  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_high_clouds**: mod09\*i: Internal CM: High Clouds  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_low_clouds**: mod09\*i: Internal CM: Low Clouds  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_snow**: mod09\*i: Internal CM: Snow  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_fire**: mod09\*i: Internal CM: Fire  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_sun_glint**: mod09\*i: Internal CM: Sun Glint  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_dust**: mod09\*i: Internal CM: Dust  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_cloud_shadow**: mod09\*i: Internal CM: Cloud Shadow  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_pixel_is_adjacent_to_cloud**: mod09\*i: Internal CM: Pixel is Adjacent to Cloud  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_cirrus**: mod09\*i: Internal CM: Cirrus  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_pan_flag**: mod09\*i: Internal CM: Pan Flag  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_criteria_for_aerosol_retrieval**: mod09\*i: Internal CM: Criteria for Aerosol Retrieval  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_aot_has_clim_val**: mod09\*i: Internal CM: AOT (aerosol optical depth) has clim. val.  
    &nbsp;&nbsp;&nbsp;&nbsp;**vi_usefulness**: mod13Q1: Quality estimation of the pixel  
    &nbsp;&nbsp;&nbsp;&nbsp;**mixed_clouds**: mod13Q1: if pixel mixed with clouds  
    &nbsp;&nbsp;&nbsp;&nbsp;**possible_snow_ice**: mod13Q1: if snow/ice present in pixel  
    &nbsp;&nbsp;&nbsp;&nbsp;**possible_shadow**: mod13Q1: if shadow is present in pixel  
    &nbsp;&nbsp;&nbsp;&nbsp;**platform**: mcd43B2: Quality of BRDF correction performed  
    &nbsp;&nbsp;&nbsp;&nbsp;**sun_z_angle_at_local_noon**: mcd43B2: Quality of BRDF correction performed  
    **band**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Band number of MODIS product (mod09Q1=[1,2],mod09A1=[1-7],m[o/y]d09GA=[1-7],m[o/y]d09CMG=[1-7], mcd43B2q=[1-7])  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1, 2, 3, 4, 5, 6, 7*  
    &nbsp;&nbsp;&nbsp;&nbsp;**1**: Band 1: Red  
    &nbsp;&nbsp;&nbsp;&nbsp;**2**: Band 2: NIR  
    &nbsp;&nbsp;&nbsp;&nbsp;**3**: Band 3: Blue  
    &nbsp;&nbsp;&nbsp;&nbsp;**4**: Band 4: Green  
    &nbsp;&nbsp;&nbsp;&nbsp;**5**: Band 5: SWIR 1  
    &nbsp;&nbsp;&nbsp;&nbsp;**6**: Band 6: SWIR 2  
    &nbsp;&nbsp;&nbsp;&nbsp;**7**: Band 7: SWIR 3  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input surface reflectance QC layer [bit array]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output QC type classification layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **productname** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of MODIS product type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *mod09Q1, mod09A1, mod09A1s, mod09GA, mod09GAs, mod09CMG, mod09CMGs, mod09CMGi, mod11A1, mod11A2, mod13A2, mcd43B2, mcd43B2q, mod13Q1*  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09Q1**: surf. refl. 250m 8-days  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09A1**: surf. refl. 500m 8-days  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09A1s**: surf. refl. 500m 8-days, State QA  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09GA**: surf. refl. 500m daily  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09GAs**: surf. refl. 500m daily, State QA  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09CMG**: surf. refl. 5000m daily  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09CMGs**: surf. refl. 5000m daily, State QA  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09CMGi**: surf. refl. 5000m daily, Internal Climatology  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod11A1**: LST 1km daily (Day/Night)  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod11A2**: LST 1km 8-days (Day/Night)  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod13A2**: VI 1km 16-days  
    &nbsp;&nbsp;&nbsp;&nbsp;**mcd43B2**: Brdf-Albedo Quality (Ancillary SDS) 1km 8-days  
    &nbsp;&nbsp;&nbsp;&nbsp;**mcd43B2q**: Brdf-Albedo Quality (BRDF SDS) 1km 8-days  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod13Q1**: VI 250m 16-days  
    **qcname** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of QC type to extract  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *adjcorr, atcorr, cloud, data_quality, diff_orbit_from_500m, modland_qa, mandatory_qa_11A1, data_quality_flag_11A1, emis_error_11A1, lst_error_11A1, data_quality_flag_11A2, emis_error_11A2, mandatory_qa_11A2, lst_error_11A2, aerosol_quantity, brdf_correction_performed, cirrus_detected, cloud_shadow, cloud_state, internal_cloud_algorithm, internal_fire_algorithm, internal_snow_mask, land_water, mod35_snow_ice, pixel_adjacent_to_cloud, salt_pan, icm_cloudy, icm_clear, icm_high_clouds, icm_low_clouds, icm_snow, icm_fire, icm_sun_glint, icm_dust, icm_cloud_shadow, icm_pixel_is_adjacent_to_cloud, icm_cirrus, icm_pan_flag, icm_criteria_for_aerosol_retrieval, icm_aot_has_clim_val, modland_qa, vi_usefulness, aerosol_quantity, pixel_adjacent_to_cloud, brdf_correction_performed, mixed_clouds, land_water, possible_snow_ice, possible_shadow, platform, land_water, sun_z_angle_at_local_noon, brdf_correction_performed, modland_qa, vi_usefulness, aerosol_quantity, pixel_adjacent_to_cloud, brdf_correction_performed, mixed_clouds, land_water, possible_snow_ice, possible_shadow*  
    &nbsp;&nbsp;&nbsp;&nbsp;**adjcorr**: mod09: Adjacency Correction  
    &nbsp;&nbsp;&nbsp;&nbsp;**atcorr**: mod09: Atmospheric Correction  
    &nbsp;&nbsp;&nbsp;&nbsp;**cloud**: mod09: Cloud State  
    &nbsp;&nbsp;&nbsp;&nbsp;**data_quality**: mod09: Band-Wise Data Quality Flag  
    &nbsp;&nbsp;&nbsp;&nbsp;**diff_orbit_from_500m**: mod09: 250m Band is at Different Orbit than 500m  
    &nbsp;&nbsp;&nbsp;&nbsp;**modland_qa**: mod13Q1: MODIS Land General Quality Assessment  
    &nbsp;&nbsp;&nbsp;&nbsp;**mandatory_qa_11A1**: mod11A1: MODIS Land General Quality Assessment  
    &nbsp;&nbsp;&nbsp;&nbsp;**data_quality_flag_11A1**: mod11A1: Detailed Quality Indications  
    &nbsp;&nbsp;&nbsp;&nbsp;**emis_error_11A1**: mod11A1: Average Emissivity Error Classes  
    &nbsp;&nbsp;&nbsp;&nbsp;**lst_error_11A1**: mod11A1: Average LST Error Classes  
    &nbsp;&nbsp;&nbsp;&nbsp;**data_quality_flag_11A2**: mod11A2: Detailed Quality Indications  
    &nbsp;&nbsp;&nbsp;&nbsp;**emis_error_11A2**: mod11A2: Average Emissivity Error Classes  
    &nbsp;&nbsp;&nbsp;&nbsp;**mandatory_qa_11A2**: mod11A2: MODIS Land General Quality Assessment  
    &nbsp;&nbsp;&nbsp;&nbsp;**lst_error_11A2**: mod11A2: Average LST Error Classes  
    &nbsp;&nbsp;&nbsp;&nbsp;**aerosol_quantity**: mod13Q1: Quantity range of Aerosol  
    &nbsp;&nbsp;&nbsp;&nbsp;**brdf_correction_performed**: mod13Q1: if BRDF correction performed  
    &nbsp;&nbsp;&nbsp;&nbsp;**cirrus_detected**: mod09\*s: StateQA Cirrus Detected  
    &nbsp;&nbsp;&nbsp;&nbsp;**cloud_shadow**: mod09\*s: StateQA Cloud Shadow  
    &nbsp;&nbsp;&nbsp;&nbsp;**cloud_state**: mod09\*s: StateQA Cloud State  
    &nbsp;&nbsp;&nbsp;&nbsp;**internal_cloud_algorithm**: mod09\*s: StateQA Internal Cloud Algorithm  
    &nbsp;&nbsp;&nbsp;&nbsp;**internal_fire_algorithm**: mod09\*s: StateQA Internal Fire Algorithm  
    &nbsp;&nbsp;&nbsp;&nbsp;**internal_snow_mask**: mod09\*s: StateQA Internal Snow Mask  
    &nbsp;&nbsp;&nbsp;&nbsp;**land_water**: mod13Q1: separate land from various water objects  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod35_snow_ice**: mod09\*s: StateQA mod35 Snow Ice  
    &nbsp;&nbsp;&nbsp;&nbsp;**pixel_adjacent_to_cloud**: mod13Q1: if pixel is a cloud neighbour  
    &nbsp;&nbsp;&nbsp;&nbsp;**salt_pan**: mod09\*s: StateQA Salt Pan (mod09GAs)  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_cloudy**: mod09\*i: Internal CM: Cloudy  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_clear**: mod09\*i: Internal CM: Clear  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_high_clouds**: mod09\*i: Internal CM: High Clouds  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_low_clouds**: mod09\*i: Internal CM: Low Clouds  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_snow**: mod09\*i: Internal CM: Snow  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_fire**: mod09\*i: Internal CM: Fire  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_sun_glint**: mod09\*i: Internal CM: Sun Glint  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_dust**: mod09\*i: Internal CM: Dust  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_cloud_shadow**: mod09\*i: Internal CM: Cloud Shadow  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_pixel_is_adjacent_to_cloud**: mod09\*i: Internal CM: Pixel is Adjacent to Cloud  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_cirrus**: mod09\*i: Internal CM: Cirrus  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_pan_flag**: mod09\*i: Internal CM: Pan Flag  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_criteria_for_aerosol_retrieval**: mod09\*i: Internal CM: Criteria for Aerosol Retrieval  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_aot_has_clim_val**: mod09\*i: Internal CM: AOT (aerosol optical depth) has clim. val.  
    &nbsp;&nbsp;&nbsp;&nbsp;**vi_usefulness**: mod13Q1: Quality estimation of the pixel  
    &nbsp;&nbsp;&nbsp;&nbsp;**mixed_clouds**: mod13Q1: if pixel mixed with clouds  
    &nbsp;&nbsp;&nbsp;&nbsp;**possible_snow_ice**: mod13Q1: if snow/ice present in pixel  
    &nbsp;&nbsp;&nbsp;&nbsp;**possible_shadow**: mod13Q1: if shadow is present in pixel  
    &nbsp;&nbsp;&nbsp;&nbsp;**platform**: mcd43B2: Quality of BRDF correction performed  
    &nbsp;&nbsp;&nbsp;&nbsp;**sun_z_angle_at_local_noon**: mcd43B2: Quality of BRDF correction performed  
    **band** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Band number of MODIS product (mod09Q1=[1,2],mod09A1=[1-7],m[o/y]d09GA=[1-7],m[o/y]d09CMG=[1-7], mcd43B2q=[1-7])  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1, 2, 3, 4, 5, 6, 7*  
    &nbsp;&nbsp;&nbsp;&nbsp;**1**: Band 1: Red  
    &nbsp;&nbsp;&nbsp;&nbsp;**2**: Band 2: NIR  
    &nbsp;&nbsp;&nbsp;&nbsp;**3**: Band 3: Blue  
    &nbsp;&nbsp;&nbsp;&nbsp;**4**: Band 4: Green  
    &nbsp;&nbsp;&nbsp;&nbsp;**5**: Band 5: SWIR 1  
    &nbsp;&nbsp;&nbsp;&nbsp;**6**: Band 6: SWIR 2  
    &nbsp;&nbsp;&nbsp;&nbsp;**7**: Band 7: SWIR 3  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input surface reflectance QC layer [bit array]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output QC type classification layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **productname** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of MODIS product type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *mod09Q1, mod09A1, mod09A1s, mod09GA, mod09GAs, mod09CMG, mod09CMGs, mod09CMGi, mod11A1, mod11A2, mod13A2, mcd43B2, mcd43B2q, mod13Q1*  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09Q1**: surf. refl. 250m 8-days  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09A1**: surf. refl. 500m 8-days  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09A1s**: surf. refl. 500m 8-days, State QA  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09GA**: surf. refl. 500m daily  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09GAs**: surf. refl. 500m daily, State QA  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09CMG**: surf. refl. 5000m daily  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09CMGs**: surf. refl. 5000m daily, State QA  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod09CMGi**: surf. refl. 5000m daily, Internal Climatology  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod11A1**: LST 1km daily (Day/Night)  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod11A2**: LST 1km 8-days (Day/Night)  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod13A2**: VI 1km 16-days  
    &nbsp;&nbsp;&nbsp;&nbsp;**mcd43B2**: Brdf-Albedo Quality (Ancillary SDS) 1km 8-days  
    &nbsp;&nbsp;&nbsp;&nbsp;**mcd43B2q**: Brdf-Albedo Quality (BRDF SDS) 1km 8-days  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod13Q1**: VI 250m 16-days  
    **qcname** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of QC type to extract  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *adjcorr, atcorr, cloud, data_quality, diff_orbit_from_500m, modland_qa, mandatory_qa_11A1, data_quality_flag_11A1, emis_error_11A1, lst_error_11A1, data_quality_flag_11A2, emis_error_11A2, mandatory_qa_11A2, lst_error_11A2, aerosol_quantity, brdf_correction_performed, cirrus_detected, cloud_shadow, cloud_state, internal_cloud_algorithm, internal_fire_algorithm, internal_snow_mask, land_water, mod35_snow_ice, pixel_adjacent_to_cloud, salt_pan, icm_cloudy, icm_clear, icm_high_clouds, icm_low_clouds, icm_snow, icm_fire, icm_sun_glint, icm_dust, icm_cloud_shadow, icm_pixel_is_adjacent_to_cloud, icm_cirrus, icm_pan_flag, icm_criteria_for_aerosol_retrieval, icm_aot_has_clim_val, modland_qa, vi_usefulness, aerosol_quantity, pixel_adjacent_to_cloud, brdf_correction_performed, mixed_clouds, land_water, possible_snow_ice, possible_shadow, platform, land_water, sun_z_angle_at_local_noon, brdf_correction_performed, modland_qa, vi_usefulness, aerosol_quantity, pixel_adjacent_to_cloud, brdf_correction_performed, mixed_clouds, land_water, possible_snow_ice, possible_shadow*  
    &nbsp;&nbsp;&nbsp;&nbsp;**adjcorr**: mod09: Adjacency Correction  
    &nbsp;&nbsp;&nbsp;&nbsp;**atcorr**: mod09: Atmospheric Correction  
    &nbsp;&nbsp;&nbsp;&nbsp;**cloud**: mod09: Cloud State  
    &nbsp;&nbsp;&nbsp;&nbsp;**data_quality**: mod09: Band-Wise Data Quality Flag  
    &nbsp;&nbsp;&nbsp;&nbsp;**diff_orbit_from_500m**: mod09: 250m Band is at Different Orbit than 500m  
    &nbsp;&nbsp;&nbsp;&nbsp;**modland_qa**: mod13Q1: MODIS Land General Quality Assessment  
    &nbsp;&nbsp;&nbsp;&nbsp;**mandatory_qa_11A1**: mod11A1: MODIS Land General Quality Assessment  
    &nbsp;&nbsp;&nbsp;&nbsp;**data_quality_flag_11A1**: mod11A1: Detailed Quality Indications  
    &nbsp;&nbsp;&nbsp;&nbsp;**emis_error_11A1**: mod11A1: Average Emissivity Error Classes  
    &nbsp;&nbsp;&nbsp;&nbsp;**lst_error_11A1**: mod11A1: Average LST Error Classes  
    &nbsp;&nbsp;&nbsp;&nbsp;**data_quality_flag_11A2**: mod11A2: Detailed Quality Indications  
    &nbsp;&nbsp;&nbsp;&nbsp;**emis_error_11A2**: mod11A2: Average Emissivity Error Classes  
    &nbsp;&nbsp;&nbsp;&nbsp;**mandatory_qa_11A2**: mod11A2: MODIS Land General Quality Assessment  
    &nbsp;&nbsp;&nbsp;&nbsp;**lst_error_11A2**: mod11A2: Average LST Error Classes  
    &nbsp;&nbsp;&nbsp;&nbsp;**aerosol_quantity**: mod13Q1: Quantity range of Aerosol  
    &nbsp;&nbsp;&nbsp;&nbsp;**brdf_correction_performed**: mod13Q1: if BRDF correction performed  
    &nbsp;&nbsp;&nbsp;&nbsp;**cirrus_detected**: mod09\*s: StateQA Cirrus Detected  
    &nbsp;&nbsp;&nbsp;&nbsp;**cloud_shadow**: mod09\*s: StateQA Cloud Shadow  
    &nbsp;&nbsp;&nbsp;&nbsp;**cloud_state**: mod09\*s: StateQA Cloud State  
    &nbsp;&nbsp;&nbsp;&nbsp;**internal_cloud_algorithm**: mod09\*s: StateQA Internal Cloud Algorithm  
    &nbsp;&nbsp;&nbsp;&nbsp;**internal_fire_algorithm**: mod09\*s: StateQA Internal Fire Algorithm  
    &nbsp;&nbsp;&nbsp;&nbsp;**internal_snow_mask**: mod09\*s: StateQA Internal Snow Mask  
    &nbsp;&nbsp;&nbsp;&nbsp;**land_water**: mod13Q1: separate land from various water objects  
    &nbsp;&nbsp;&nbsp;&nbsp;**mod35_snow_ice**: mod09\*s: StateQA mod35 Snow Ice  
    &nbsp;&nbsp;&nbsp;&nbsp;**pixel_adjacent_to_cloud**: mod13Q1: if pixel is a cloud neighbour  
    &nbsp;&nbsp;&nbsp;&nbsp;**salt_pan**: mod09\*s: StateQA Salt Pan (mod09GAs)  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_cloudy**: mod09\*i: Internal CM: Cloudy  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_clear**: mod09\*i: Internal CM: Clear  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_high_clouds**: mod09\*i: Internal CM: High Clouds  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_low_clouds**: mod09\*i: Internal CM: Low Clouds  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_snow**: mod09\*i: Internal CM: Snow  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_fire**: mod09\*i: Internal CM: Fire  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_sun_glint**: mod09\*i: Internal CM: Sun Glint  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_dust**: mod09\*i: Internal CM: Dust  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_cloud_shadow**: mod09\*i: Internal CM: Cloud Shadow  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_pixel_is_adjacent_to_cloud**: mod09\*i: Internal CM: Pixel is Adjacent to Cloud  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_cirrus**: mod09\*i: Internal CM: Cirrus  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_pan_flag**: mod09\*i: Internal CM: Pan Flag  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_criteria_for_aerosol_retrieval**: mod09\*i: Internal CM: Criteria for Aerosol Retrieval  
    &nbsp;&nbsp;&nbsp;&nbsp;**icm_aot_has_clim_val**: mod09\*i: Internal CM: AOT (aerosol optical depth) has clim. val.  
    &nbsp;&nbsp;&nbsp;&nbsp;**vi_usefulness**: mod13Q1: Quality estimation of the pixel  
    &nbsp;&nbsp;&nbsp;&nbsp;**mixed_clouds**: mod13Q1: if pixel mixed with clouds  
    &nbsp;&nbsp;&nbsp;&nbsp;**possible_snow_ice**: mod13Q1: if snow/ice present in pixel  
    &nbsp;&nbsp;&nbsp;&nbsp;**possible_shadow**: mod13Q1: if shadow is present in pixel  
    &nbsp;&nbsp;&nbsp;&nbsp;**platform**: mcd43B2: Quality of BRDF correction performed  
    &nbsp;&nbsp;&nbsp;&nbsp;**sun_z_angle_at_local_noon**: mcd43B2: Quality of BRDF correction performed  
    **band** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Band number of MODIS product (mod09Q1=[1,2],mod09A1=[1-7],m[o/y]d09GA=[1-7],m[o/y]d09CMG=[1-7], mcd43B2q=[1-7])  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1, 2, 3, 4, 5, 6, 7*  
    &nbsp;&nbsp;&nbsp;&nbsp;**1**: Band 1: Red  
    &nbsp;&nbsp;&nbsp;&nbsp;**2**: Band 2: NIR  
    &nbsp;&nbsp;&nbsp;&nbsp;**3**: Band 3: Blue  
    &nbsp;&nbsp;&nbsp;&nbsp;**4**: Band 4: Green  
    &nbsp;&nbsp;&nbsp;&nbsp;**5**: Band 5: SWIR 1  
    &nbsp;&nbsp;&nbsp;&nbsp;**6**: Band 6: SWIR 2  
    &nbsp;&nbsp;&nbsp;&nbsp;**7**: Band 7: SWIR 3  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*i.modis.qc* extracts Requested Quality Assessment flags from the
following MODIS products: MOD09A1, MOD09Q1, MOD11A1, MOD11A2, MOD13A2,
MOD13Q1, MCD43B2. This does include MOD09A1 QA_state_500m layer (see
Notes).  
Added MOD09GA support in 2016, it follows MOD09A1 and its StateQA, but
does not have BRDF State QA, instead has Salt Pan State QA.

### MOD09A1 and MOD09Q1

```sh
MOD09A1/Q1: MODLAND QA Bits. bits=[0-1]
```

- \[00\]= class 0: Corrected product produced at ideal quality -- all
  bands
- \[01\]= class 1: Corrected product produced at less than ideal quality
  -- some or all bands
- \[10\]= class 2: Corrected product NOT produced due to cloud effect --
  all bands
- \[11\]= class 3: Corrected product NOT produced due to other reasons
  -- some or all bands maybe be fill value (Note that a value of \[11\]
  overrides a value of \[01\])

```sh
MOD09Q1: Cloud State. bits=[2-3] 
```

- \[00\]= class 0: Clear -- No clouds
- \[01\]= class 1: Cloudy
- \[10\]= class 2: Mixed
- \[11\]= class 3: Not Set ; Assumed Clear

```sh
MOD09Q1: Band-wise Data Quality 250m bits=[4-7][8-11]
MOD09A1: Band-wise Data Quality 500m bits=[2-5][6-9][10-13][14-17][18-21][22-25][26-29]
```

- \[0000\]= class 0: highest quality
- \[0111\]= class 1: noisy detector
- \[1000\]= class 2: dead detector; data interpolated in L1B
- \[1001\]= class 3: solar zenith ≥ 86 degrees
- \[1010\]= class 4: solar zenith ≥ 85 and \< 86 degrees
- \[1011\]= class 5: missing input
- \[1100\]= class 6: internal constant used in place of climatological
  data for at least one atmospheric constant
- \[1101\]= class 7: correction out of bounds, pixel constrained to
  extreme allowable value
- \[1110\]= class 8: L1B data faulty
- \[1111\]= class 9: not processed due to deep ocean or cloud
- Class 10-15: Combination of bits unused

```sh
MOD09A1/Q1: Atmospheric correction bit=[12]/[30]
```

- \[0\]= class 0: Not Corrected product
- \[1\]= class 1: Corrected product

```sh
MOD09A1/Q1: Adjacency correction bit=[13]/[31]
```

- \[0\]= class 0: Not Corrected product
- \[1\]= class 1: Corrected product

```sh
MOD09Q1: Different orbit from 500m product, bit=[14]
```

- \[0\]= class 0: same orbit as 500m
- \[1\]= class 1: different orbit from 500m

```sh
MOD09A1s: Cloud State bits=[0-1]
```

- \[00\]= class 0: clear
- \[01\]= class 1: cloudy
- \[10\]= class 2: mixed
- \[11\]= class 3: not set, assumed clear

```sh
MOD09A1s: Cloud shadow bits=[2]
```

- \[0\]= class 0: no
- \[1\]= class 1: yes

```sh
MOD09A1s: Land/Water Flag bits=[3-5]
```

- \[000\]= class 0: Shallow ocean
- \[001\]= class 1: Land
- \[010\]= class 2: Ocean coastlines and lake shorelines
- \[011\]= class 3: Shallow inland water
- \[100\]= class 4: Ephemeral water
- \[101\]= class 5: Deep inland water
- \[110\]= class 6: Continental/moderate ocean
- \[111\]= class 7: Deep ocean

```sh
MOD09A1s: Aerosol Quantity bits=[6-7]
```

- \[00\]= class 0: Climatology
- \[01\]= class 1: Low
- \[10\]= class 2: Average
- \[11\]= class 3: High

```sh
MOD09A1s: Cirrus detected bits=[8-9]
```

- \[00\]= class 0: None
- \[01\]= class 1: Small
- \[10\]= class 2: Average
- \[11\]= class 3: High

```sh
MOD09A1s: Internal Cloud Algorithm Flag bits=[10]
```

- \[0\]= class 0: No cloud
- \[1\]= class 1: Cloud

```sh
MOD09A1s: Internal Fire Algorithm Flag bits=[11]
```

- \[0\]= class 0: No fire
- \[1\]= class 1: Fire

```sh
MOD09A1s: MOD35 snow/ice flag bits=[12]
```

- \[0\]= class 0: No
- \[1\]= class 1: Yes

```sh
MOD09A1s: Pixel adjacent to cloud bits=[13]
```

- \[0\]= class 0: No
- \[1\]= class 1: Yes

```sh
MOD09A1s: BRDF correction performed bits=[14]
```

- \[0\]= class 0: No
- \[1\]= class 1: Yes

```sh
MOD09A1s: Internal Snow Mask bits=[15]
```

- \[0\]= class 0: No snow
- \[1\]= class 1: Snow

### MOD11A1

```sh
MOD11A1: Mandatory QA Flags bits=[0-1]
```

- \[00\]= class 0: LST produced, good quality, not necessary to examine
  more detailed QA
- \[01\]= class 1: LST produced, other quality, recommend examination of
  more detailed QA
- \[10\]= class 2: LST not produced due to cloud effects
- \[11\]= class 3: LST not produced primarily due to reasons other than
  cloud

```sh
MOD11A1: Data Quality Flag bits=[2-3]
```

- \[00\]= class 0: Good data quality of L1B in bands 31 and 32
- \[01\]= class 1: Other quality data
- \[10\]= class 2: TBD
- \[11\]= class 3: TBD

```sh
MOD11A1: Emis Error Flag bits=[4-5]
```

- \[00\]= class 0: Average emissivity error ≤ 0.01
- \[01\]= class 1: Average emissivity error ≤ 0.02
- \[10\]= class 2: Average emissivity error ≤ 0.04
- \[11\]= class 3: Average emissivity error \> 0.04

```sh
MOD11A1: LST Error Flag bits=[6-7]
```

- \[00\]= class 0: Average LST error ≤ 1
- \[01\]= class 1: Average LST error ≤ 2
- \[10\]= class 2: Average LST error ≤ 3
- \[11\]= class 3: Average LST error \> 3

### MOD11A2

```sh
MOD11A2: Mandatory QA Flags bits=[0-1]
```

- \[00\]= class 0: LST produced, good quality, not necessary to examine
  more detailed QA
- \[01\]= class 1: LST produced, other quality, recommend examination of
  more detailed QA
- \[10\]= class 2: LST not produced due to cloud effects
- \[11\]= class 3: LST not produced primarily due to reasons other than
  cloud

```sh
MOD11A2: Data Quality Flag bits=[2-3]
```

- \[00\]= class 0: Good data quality of L1B in 7 TIR bands
- \[01\]= class 1: Other quality data
- \[10\]= class 2: TBD
- \[11\]= class 3: TBD

```sh
MOD11A2: Emis Error Flag bits=[4-5]
```

- \[00\]= class 0: Average emissivity error ≤ 0.01
- \[01\]= class 1: Average emissivity error ≤ 0.02
- \[10\]= class 2: Average emissivity error ≤ 0.04
- \[11\]= class 3: Average emissivity error \> 0.04

```sh
MOD11A2: LST Error Flag bits=[6-7]
```

- \[00\]= class 0: Average LST error ≤ 1
- \[01\]= class 1: Average LST error ≤ 2
- \[10\]= class 2: Average LST error ≤ 3
- \[11\]= class 3: Average LST error \> 3

### MOD13A2

```sh
MOD13A2: Mandatory QA Flags 1km bits[0-1]
```

- \[00\]= class 0: VI produced, good quality
- \[01\]= class 1: VI produced, but check other QA
- \[10\]= class 2: Pixel produced, but most probably cloud
- \[11\]= class 3: Pixel not produced due to other reasons than clouds

```sh
MOD13A2: VI Usefulness Flag bits[2-5]
```

- \[0000\]= class 0: Highest quality
- \[0001\]= class 1: Lower quality
- \[0010\]= class 2: Decreasing quality
- \[0100\]= class 3: Decreasing quality
- \[1000\]= class 4: Decreasing quality
- \[1001\]= class 5: Decreasing quality
- \[1010\]= class 6: Decreasing quality
- \[1100\]= class 7: Lowest quality
- \[1101\]= class 8: Quality so low that it is not useful
- \[1110\]= class 9: L1B data faulty
- \[1111\]= class 10: Not useful for any other reason/not processed

```sh
MOD13A2: Aerosol quantity Flags 1km bits[6-7]
```

- \[00\]= class 0: Climatology
- \[01\]= class 1: Low
- \[10\]= class 2: Average
- \[11\]= class 3: High

```sh
MOD13A2: Adjacent cloud detected 1km bit[8]
```

- \[00\]= class 0: No
- \[01\]= class 1: Yes

```sh
MOD13A2: Atmosphere BRDF correction performed 1km bit[9]
```

- \[00\]= class 0: No
- \[01\]= class 1: Yes

```sh
MOD13A2: Mixed clouds 1km bit[10]
```

- \[00\]= class 0: No
- \[01\]= class 1: Yes

```sh
MOD13A2: Land/Water Flags 1km bits[11-13]
```

- \[000\]= class 0: Shallow Ocean
- \[001\]= class 1: Land (Nothing else but land)
- \[010\]= class 2: Ocean Coastlines and lake shorelines
- \[011\]= class 3: Shallow inland water
- \[100\]= class 4: Ephemeral water
- \[101\]= class 5: Deep inland water
- \[110\]= class 6: Moderate or continental ocean
- \[111\]= class 7: Deep ocean

```sh
MOD13A2: Possible Snow/Ice 1km bits[14]
```

- \[0\]= class 0: No
- \[1\]= class 1: Yes

```sh
MOD13A2: Possible Shadow 1km bits[15]
```

- \[0\]= class 0: No
- \[1\]= class 1: Yes

### MOD13Q1

```sh
MOD13Q1: Mandatory QA Flags 250m bits[0-1]
```

- \[00\]= class 0: VI produced, good quality
- \[01\]= class 1: VI produced, but check other QA
- \[10\]= class 2: Pixel produced, but most probably cloud
- \[11\]= class 3: Pixel not produced due to other reasons than clouds

```sh
MOD13Q1: VI Usefulness Flag 250m bits[2-5]
```

- \[0000\]= class 0: Highest quality
- \[0001\]= class 1: Lower quality
- \[0010\]= class 2: Decreasing quality
- \[0100\]= class 3: Decreasing quality
- \[1000\]= class 4: Decreasing quality
- \[1001\]= class 5: Decreasing quality
- \[1010\]= class 6: Decreasing quality
- \[1100\]= class 7: Lowest quality
- \[1101\]= class 8: Quality so low that it is not useful
- \[1110\]= class 9: L1B data faulty
- \[1111\]= class 10: Not useful for any other reason/not processed

```sh
MOD13Q1: Aerosol quantity Flags 250m bits[6-7]
```

- \[00\]= class 0: Climatology
- \[01\]= class 1: Low
- \[10\]= class 2: Average
- \[11\]= class 3: High

```sh
MOD13Q1: Adjacent cloud detected 250m bit[8]
```

- \[00\]= class 0: No
- \[01\]= class 1: Yes

```sh
MOD13Q1: Atmosphere BRDF correction performed 250m bit[9]
```

- \[00\]= class 0: No
- \[01\]= class 1: Yes

```sh
MOD13Q1: Mixed clouds 250m bit[10]
```

- \[00\]= class 0: No
- \[01\]= class 1: Yes

```sh
MOD13Q1: Land/Water Flags 250m bits[11-13]
```

- \[000\]= class 0: Shallow Ocean
- \[001\]= class 1: Land (Nothing else but land)
- \[010\]= class 2: Ocean Coastlines and lake shorelines
- \[011\]= class 3: Shallow inland water
- \[100\]= class 4: Ephemeral water
- \[101\]= class 5: Deep inland water
- \[110\]= class 6: Moderate or continental ocean
- \[111\]= class 7: Deep ocean

```sh
MOD13Q1: Possible Snow/Ice 250m bits[14]
```

- \[0\]= class 0: No
- \[1\]= class 1: Yes

```sh
MOD13Q1: Possible Shadow 250m bits[15]
```

- \[0\]= class 0: No
- \[1\]= class 1: Yes

### MCD43B2

```sh
MCD43B2: Albedo Quality Ancillary Platform Data 1km bits[0-3]
SDS: BRDF_Albedo_Ancillary
```

- \[0000\]= class 0: Satellite Platform: Terra
- \[0001\]= class 1: Satellite Platform: Terrra/Aqua
- \[0010\]= class 2: Satellite Platform: Aqua
- \[1111\]= class 15: Fill Value
- Classes 3-14: Not used

```sh
MCD43B2: Albedo Quality Ancillary Land/Water Data 1km bits[4-7]
SDS: BRDF_Albedo_Ancillary
```

- \[0000\] class 0: Shallow Ocean
- \[0001\] class 1: Land (Nothing else but land)
- \[0010\] class 2: Ocean and lake shorelines
- \[0011\] class 3: Shallow inland water
- \[0100\] class 4: Ephemeral water
- \[0101\] class 5: Deep inland water
- \[0110\] class 6: Moderate or continental ocean
- \[0111\] class 7: Deep ocean
- \[1111\] class 15: Fill Value
- Classes 8-14: Not used

```sh
MCD43B2: Albedo Quality Ancillary Sun Zenith Angle at Local Solar Noon Data 1km bits[8-14]
SDS: BRDF_Albedo_Ancillary
```

```sh
MCD43B2: Band-wise Albedo Quality Data 1km
SDS: BRDF_Albedo_Band_Quality
```

bits\[0-3\]\[4-7\]\[8-11\]\[12-15\]\[16-19\]\[20-23\]\[24-27\]  

- \[0000\]= class 0: best quality, 75% or more with best full inversions
- \[0001\]= class 1: good quality, 75% or more with full inversions
- \[0010\]= class 2: Mixed, 50% or less full inversions and 25% or less
  fill values
- \[0011\]= class 3: All magnitude inversions or 50% or less fill values
- \[0100\]= class 4: 75% or more fill values
- Classes 5-14: Not Used
- \[1111\]= class 15: Fill Value

## NOTES

In MOD09A1: It seems that cloud related info is not filled properly in
the standard QC (MOD09A1 in this module) since version 3, State-QA 500m
images (MOD09A1s in this module) should be used (see Vermote et al.,
2008). MOD11A2 quality control (QC) bands do not have a FillValue
(No-data) according to [MODIS Land Products
site](https://lpdaac.usgs.gov/dataset_discovery/modis/modis_products_table/mod11a2_v006).
However, the metadata of the QC bands (i.e.: `gdalinfo QC_band`) shows
`No-data=0`. This value is then transformed into GRASS NULLs when data
is imported through [r.in.gdal](r.in.gdal.md). Applying *i.modis.qc* on
those QC bands will not give the expected range of values in the
different QC bits. Therefore, before using *i.modis.qc*, the user needs
to set the NULL value in QC bands back to zero (i.e.:
`r.null map=QC_band null=0`) or just edit the metadata with GDAL
utilities before importing into GRASS. This is a known issue for
MOD11A2 (8-day LST product), but other MODIS products might be affected
as well.

## TODO

Add more daily products.

## REFERENCES

- [MODIS
  Products](https://lpdaac.usgs.gov/dataset_discovery/modis/modis_products_table)
- Vermote E.F., Kotchenova S.Y., Ray J.P. MODIS Surface Reflectance
  User's Guide. Version 1.2. June 2008. MODIS Land Surface Reflectance
  Science Computing Facility. [Homepage](http://modis-sr.ltdri.org)

## SEE ALSO

*[i.vi](i.vi.md)*

## AUTHOR

Yann Chemin

## SOURCE CODE

Available at: [i.modis.qc source code](https://github.com/OSGeo/grass/tree/main/imagery/i.modis.qc)
([history](https://github.com/OSGeo/grass/commits/main/imagery/i.modis.qc))  
Latest change: Friday Aug 08 09:39:19 2025 in commit [ed72c71](https://github.com/OSGeo/grass/commit/ed72c71d8580e872cd6fd345c84a15c324764465)
