---
name: t.rast.what.py
description: Sample a space time raster dataset at specific vector point coordinates and write the output to stdout using different layouts
keywords: [ temporal, sampling, raster, time ]
---

# t.rast.what.py

Sample a space time raster dataset at specific vector point coordinates and write the output to stdout using different layouts

=== "Command line"

    **t.rast.what.py**
    [**-niv**]
    [**points**=*name*]
    [**coordinates**=*east,north*]
    **strds**=*name*
    [**output**=*name*]
    [**where**=*sql_query*]
    [**null_value**=*string*]
    [**separator**=*character*]
    [**order**=*string* [,*string*,...]]
    [**layout**=*string*]
    [**nprocs**=*integer*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    t.rast.what.py strds=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***t.rast.what.py***",
        **points**=*None*,
        **coordinates**=*None*,
        **strds**,
        **output**=*"-"*,
        **where**=*None*,
        **null_value**=*None*,
        **separator**=*"pipe"*,
        **order**=*"start_time"*,
        **layout**=*"row"*,
        **nprocs**=*1*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("t.rast.what.py", strds="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.t_rast_what_py*(**points**=*None*,
        **coordinates**=*None*,
        **strds**,
        **output**=*"-"*,
        **where**=*None*,
        **null_value**=*None*,
        **separator**=*"pipe"*,
        **order**=*"start_time"*,
        **layout**=*"row"*,
        **nprocs**=*1*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.t_rast_what_py(strds="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **points**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    **coordinates**=*east,north*  
    &nbsp;&nbsp;&nbsp;&nbsp;Comma separated list of coordinates  
    **strds**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the input space time raster dataset  
    **output**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for the output file or "-" in case stdout should be used  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-*  
    **where**=*sql_query*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword used in the temporal GIS framework  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: start_time &gt; '2001-01-01 12:30:00'  
    **null_value**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;String representing NULL value  
    **separator**=*character*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *pipe*  
    **order**=*string* [,*string*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Sort the maps by category  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *id,  name,  creator,  mapset,  creation_time,  modification_time,  start_time,  end_time,  north,  south,  west,  east,  min,  max*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *start_time*  
    **layout**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;The layout of the output. One point per row (row), one point per column (col), all timsteps in one row (timerow)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *row,  col,  timerow*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *row*  
    **nprocs**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of r.what processes to run in parallel  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **-n**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output header row  
    **-i**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use stdin as input and ignore coordinates and point option  
    **-v**  
    &nbsp;&nbsp;&nbsp;&nbsp;Show the category for vector points map  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **points** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **coordinates** : tuple[float, float] | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Comma separated list of coordinates  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, coords, *east,north*  
    **strds** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the input space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, strds, *name*  
    **output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for the output file or "-" in case stdout should be used  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-*  
    **where** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword used in the temporal GIS framework  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: start_time &gt; '2001-01-01 12:30:00'  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *sql_query*  
    **null_value** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;String representing NULL value  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *string*  
    **separator** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, separator, *character*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *pipe*  
    **order** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Sort the maps by category  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *id,  name,  creator,  mapset,  creation_time,  modification_time,  start_time,  end_time,  north,  south,  west,  east,  min,  max*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *start_time*  
    **layout** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The layout of the output. One point per row (row), one point per column (col), all timsteps in one row (timerow)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *row,  col,  timerow*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *row*  
    **nprocs** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of r.what processes to run in parallel  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *n*, *i*, *v*  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Output header row  
    &nbsp;&nbsp;&nbsp;&nbsp;**i**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use stdin as input and ignore coordinates and point option  
    &nbsp;&nbsp;&nbsp;&nbsp;**v**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Show the category for vector points map  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **points** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **coordinates** : tuple[float, float] | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Comma separated list of coordinates  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, coords, *east,north*  
    **strds** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the input space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, strds, *name*  
    **output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for the output file or "-" in case stdout should be used  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-*  
    **where** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword used in the temporal GIS framework  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: start_time &gt; '2001-01-01 12:30:00'  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *sql_query*  
    **null_value** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;String representing NULL value  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *string*  
    **separator** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, separator, *character*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *pipe*  
    **order** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Sort the maps by category  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *id,  name,  creator,  mapset,  creation_time,  modification_time,  start_time,  end_time,  north,  south,  west,  east,  min,  max*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *start_time*  
    **layout** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;The layout of the output. One point per row (row), one point per column (col), all timsteps in one row (timerow)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *row,  col,  timerow*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *row*  
    **nprocs** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of r.what processes to run in parallel  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *n*, *i*, *v*  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Output header row  
    &nbsp;&nbsp;&nbsp;&nbsp;**i**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use stdin as input and ignore coordinates and point option  
    &nbsp;&nbsp;&nbsp;&nbsp;**v**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Show the category for vector points map  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*t.rast.what* is designed to sample space time raster datasets at
specific point coordinates using [r.what](r.what.md) internally. The
output of [r.what](r.what.md) is transformed to different output
layouts. The output layouts can be specified using the *layout* option.

Three layouts can be specified:

- *row* - Row order, one vector sample point value per row
- *col* - Column order, create a column for each vector sample point of
  a single time step/raster layer
- *timerow* - Time order, create a column for each time step, this order
  is the original *r.what* output, except that the column names are the
  timestamps

Please have a look at the example to see the supported layouts.

This module is designed to run several instances of *r.what* to sample
subsets of a space time raster dataset in parallel. Several intermediate
text files will be created that are merged into a single file at the end
of the processing.

Coordinates can be provided as vector map using the *points* option or
as comma separated coordinate list with the *coordinates* option.

An output file can be specified using the *output* option. Stdout will
be used if no output is specified or if the *output* option is set to
"-".

## EXAMPLES

### Data preparation

In the following examples we sample a space time raster dataset that
contains 4 raster map layers. First we create the STRDS that will be
sampled with *t.rast.what*.

```sh
g.region s=0 n=80 w=0 e=120 b=0 t=50 res=10

# Generate data
r.mapcalc expression="a_1 = 1" -s
r.mapcalc expression="a_2 = 2" -s
r.mapcalc expression="a_3 = 3" -s
r.mapcalc expression="a_4 = 4" -s

t.create type=strds output=A title="A test" descr="A test"

t.register -i type=raster input=A maps=a_1,a_2,a_3,a_4 \
    start='1990-01-01' increment="1 month"
```

### Example 1

The first approach uses text coordinates as input and stdout as output,
the layout is one coordinate(point per column:

```sh
t.rast.what strds=A coordinates="115,36,79,45" layout=col -n

start|end|115.0000000000;36.0000000000|79.0000000000;45.0000000000
1990-01-01 00:00:00|1990-02-01 00:00:00|1|1
1990-02-01 00:00:00|1990-03-01 00:00:00|2|2
1990-03-01 00:00:00|1990-04-01 00:00:00|3|3
1990-04-01 00:00:00|1990-05-01 00:00:00|4|4
```

### Example 2

A vector map layer can be used as input to sample the STRDS. All three
available layouts are demonstrated using the vector map for sampling.

```sh
# First create the vector map layer based on random points
v.random output=points n=3 seed=1

# Row layout using a text file as output
t.rast.what strds=A points=points output=result.txt layout=row -n

cat result.txt

115.0043586274|36.3593955783|1990-01-01 00:00:00|1990-02-01 00:00:00|1
115.0043586274|36.3593955783|1990-02-01 00:00:00|1990-03-01 00:00:00|2
115.0043586274|36.3593955783|1990-03-01 00:00:00|1990-04-01 00:00:00|3
115.0043586274|36.3593955783|1990-04-01 00:00:00|1990-05-01 00:00:00|4
79.6816763826|45.2391522853|1990-01-01 00:00:00|1990-02-01 00:00:00|1
79.6816763826|45.2391522853|1990-02-01 00:00:00|1990-03-01 00:00:00|2
79.6816763826|45.2391522853|1990-03-01 00:00:00|1990-04-01 00:00:00|3
79.6816763826|45.2391522853|1990-04-01 00:00:00|1990-05-01 00:00:00|4
97.4892579600|79.2347263950|1990-01-01 00:00:00|1990-02-01 00:00:00|1
97.4892579600|79.2347263950|1990-02-01 00:00:00|1990-03-01 00:00:00|2
97.4892579600|79.2347263950|1990-03-01 00:00:00|1990-04-01 00:00:00|3
97.4892579600|79.2347263950|1990-04-01 00:00:00|1990-05-01 00:00:00|4


# Column layout order using stdout as output
t.rast.what strds=A points=points layout=col -n

start|end|115.0043586274;36.3593955783|79.6816763826;45.2391522853|97.4892579600;79.2347263950
1990-01-01 00:00:00|1990-02-01 00:00:00|1|1|1
1990-02-01 00:00:00|1990-03-01 00:00:00|2|2|2
1990-03-01 00:00:00|1990-04-01 00:00:00|3|3|3
1990-04-01 00:00:00|1990-05-01 00:00:00|4|4|4

# Timerow layout, one time series per row
# using the where statement to select a subset of the STRDS
# and stdout as output
t.rast.what strds=A points=points \
    where="start_time >= '1990-03-01'" layout=timerow -n

x|y|1990-03-01 00:00:00;1990-04-01 00:00:00|1990-04-01 00:00:00;1990-05-01 00:00:00
115.004358627375|36.3593955782903|3|4
79.681676382576|45.2391522852909|3|4
97.4892579600048|79.2347263950131|3|4
```

## SEE ALSO

*[g.region](g.region.md), [r.mask](r.mask.md)
[r.neighbors](r.neighbors.md), [r.what](r.what.md), [t.info](t.info.md),
[t.rast.aggregate.ds](t.rast.aggregate.ds.md),
[t.rast.extract](t.rast.extract.md), [v.what.strds](v.what.strds.md)*

## AUTHOR

Sören Gebbert, Thünen Institute of Climate-Smart Agriculture

## SOURCE CODE

Available at: [t.rast.what source code](https://github.com/OSGeo/grass/tree/main/temporal/t.rast.what)
([history](https://github.com/OSGeo/grass/commits/main/temporal/t.rast.what))  
Latest change: Thursday Jun 26 10:38:43 2025 in commit [a7bb103](https://github.com/OSGeo/grass/commit/a7bb103537dc5179757a05844431d87018688326)
