---
name: i.evapo.time
description: Computes temporal integration of satellite ET actual (ETa) following the daily ET reference (ETo) from meteorological station(s).
keywords: [ imagery, evapotranspiration ]
---

# i.evapo.time

Computes temporal integration of satellite ET actual (ETa) following the daily ET reference (ETo) from meteorological station(s).

=== "Command line"

    **i.evapo.time**
    **eta**=*name* [,*name*,...]
    **eta_doy**=*name* [,*name*,...]
    **eto**=*name* [,*name*,...]
    **eto_doy_min**=*float*
    **start_period**=*float*
    **end_period**=*float*
    **output**=*name*
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    i.evapo.time eta=name eta_doy=name eto=name eto_doy_min=0.0 start_period=0.0 end_period=0.0 output=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***i.evapo.time***",
        **eta**,
        **eta_doy**,
        **eto**,
        **eto_doy_min**,
        **start_period**,
        **end_period**,
        **output**,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("i.evapo.time", eta="name", eta_doy="name", eto="name", eto_doy_min=0.0, start_period=0.0, end_period=0.0, output="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.i_evapo_time*(**eta**,
        **eta_doy**,
        **eto**,
        **eto_doy_min**,
        **start_period**,
        **end_period**,
        **output**,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.i_evapo_time(eta="name", eta_doy="name", eto="name", eto_doy_min=0.0, start_period=0.0, end_period=0.0, output="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **eta**=*name* [,*name*,...] **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Names of satellite ETa raster maps [mm/d or cm/d]  
    **eta_doy**=*name* [,*name*,...] **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Names of satellite ETa Day of Year (DOY) raster maps [0-400] [-]  
    **eto**=*name* [,*name*,...] **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Names of meteorological station ETo raster maps [0-400] [mm/d or cm/d]  
    **eto_doy_min**=*float* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Value of DOY for ETo first day  
    **start_period**=*float* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Value of DOY for the first day of the period studied  
    **end_period**=*float* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Value of DOY for the last day of the period studied  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **eta** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Names of satellite ETa raster maps [mm/d or cm/d]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **eta_doy** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Names of satellite ETa Day of Year (DOY) raster maps [0-400] [-]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **eto** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Names of meteorological station ETo raster maps [0-400] [mm/d or cm/d]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **eto_doy_min** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value of DOY for ETo first day  
    **start_period** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value of DOY for the first day of the period studied  
    **end_period** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value of DOY for the last day of the period studied  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **eta** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Names of satellite ETa raster maps [mm/d or cm/d]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **eta_doy** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Names of satellite ETa Day of Year (DOY) raster maps [0-400] [-]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **eto** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Names of meteorological station ETo raster maps [0-400] [mm/d or cm/d]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **eto_doy_min** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value of DOY for ETo first day  
    **start_period** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value of DOY for the first day of the period studied  
    **end_period** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value of DOY for the last day of the period studied  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*i.evapo.time* (i.evapo.time_integration) integrates ETa in time
following a reference ET (typically) from a set of meteorological
stations dataset. Inputs:

- ETa images
- ETa images DOY (Day of Year)
- ETo images
- ETo DOYmin as a single value

Method:

1. each ETa pixel is divided by the same day ETo and become ETrF
2. each ETrF pixel is multiplied by the ETo sum for the representative
    days
3. Sum all n temporal \[ETrF\*ETo_sum\] pixels to make a summed(ET) in
    \[DOYmin;DOYmax\]

representative days calculation: let assume i belongs to range
\[DOYmin;DOYmax\]

```sh
DOYbeforeETa[i] = ( DOYofETa[i] - DOYofETa[i-1] ) / 2
DOYafterETa[i] = ( DOYofETa[i+1] - DOYofETa[i] ) / 2
```

## NOTES

ETo images preparation: If you only have one meteorological station data
set, the easiest way is:

```sh
n=0
for ETo_val in Eto[1] Eto[2] ...
do
    r.mapcalc "eto$n = $ETo_val"
    `expr n = n + 1`
done
```

with Eto\[1\], Eto\[2\], etc being a simple copy and paste from your
data file of all ETo values separated by an empty space from each other.

If you have several meteorological stations data, then you need to grid
them by generating Thiessen polygons or using different interpolation
methods for each day.

For multi-year calculations, just continue incrementing DOY values above
366, it will continue working, up to maximum input of 400 satellite
images.

![Temporal integration from a weather station](i_evapo_time.png)  
*This is an example of a temporal integration from a weather station as
done by Chemin and Alexandridis (2004)*

## References

Chemin and Alexandridis, 2004. Spatial Resolution Improvement of
Seasonal Evapotranspiration for Irrigated Rice, Zhanghe Irrigation
District, Hubei Province, China. Asian Journal of Geoinformatics, Vol.
5, No. 1, September 2004
([PDF](https://ikee.lib.auth.gr/record/270217/files/Chemin-Alexandridis.pdf))

## SEE ALSO

*[i.eb.eta](i.eb.eta.md), [i.evapo.mh](i.evapo.mh.md),
[i.evapo.pt](i.evapo.pt.md), [i.evapo.pm](i.evapo.pm.md),
[r.sun](r.sun.md)*

## AUTHOR

Yann Chemin, International Rice Research Institute, The Philippines

## SOURCE CODE

Available at: [i.evapo.time source code](https://github.com/OSGeo/grass/tree/main/imagery/i.evapo.time)
([history](https://github.com/OSGeo/grass/commits/main/imagery/i.evapo.time))  
Latest change: Thursday Jun 26 21:25:27 2025 in commit [3ec2b13](https://github.com/OSGeo/grass/commit/3ec2b13945d963db69544e3fc9339fbff72f33f3)
