---
name: r3.stats
description: Generates volume statistics for 3D raster maps.
keywords: [ raster3d, statistics, voxel, volume ]
---

# r3.stats

Generates volume statistics for 3D raster maps.

=== "Command line"

    **r3.stats**
    [**-ec**]
    **input**=*name*
    [**nsteps**=*integer*]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r3.stats input=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r3.stats***",
        **input**,
        **nsteps**=*20*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r3.stats", input="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r3_stats*(**input**,
        **nsteps**=*20*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r3_stats(input="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input 3D raster map  
    **nsteps**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of subranges to collect stats from  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *20*  
    **-e**  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculate statistics based on equal value groups  
    **-c**  
    &nbsp;&nbsp;&nbsp;&nbsp;Only print cell counts  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input 3D raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster_3d, *name*  
    **nsteps** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of subranges to collect stats from  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *20*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *e*, *c*  
    &nbsp;&nbsp;&nbsp;&nbsp;**e**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate statistics based on equal value groups  
    &nbsp;&nbsp;&nbsp;&nbsp;**c**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Only print cell counts  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input 3D raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster_3d, *name*  
    **nsteps** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of subranges to collect stats from  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *20*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *e*, *c*  
    &nbsp;&nbsp;&nbsp;&nbsp;**e**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate statistics based on equal value groups  
    &nbsp;&nbsp;&nbsp;&nbsp;**c**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Only print cell counts  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r3.stats* calculates volume statistics for 3D raster maps. Volumes are
created on base of value subranges or groups of equal values. The
default number of subranges is 20 and can vary from 1 to infinite.
Larger number of subranges will result in larger memory consumption. The
volumes are calculated in qubic meters. Null values are marked with
"\*". If the flag *-e* is set, the number of subranges will be ignored.

## NOTES

As with most GRASS 3D raster map modules, *r3.stats* operates on the
cell array defined by the current 3D region settings, not the original
extent and resolution of the input map. See *g.region*.

The region setting will not effect the memory consumption of this
module. The number of subranges in case of value range calculation or
the number of equal value groups effect the memory consumption and the
calculation time. The user can expect a huge time consumption to
calculate the equal value groups (flag *-e*) if large region settings
occur for maps which have many equal value groups (\> 100000).

## EXAMPLES

### Lausanne FOSS4G 2006 3D demo dataset example

The 3D raster map *geology* has three values. Each value represents a
specific geological layer, clay = 1, sand = 2 and bedrock = 3. To
calculate the volume of each geological layer, we can use the flag *-e*
to group equal values into single volumes. *r3.stats geology -e* will
produce the following output:

```sh
  num   |        value       |     volume    |   perc  |    count
      1             1.000000   300822000.000   50.13700        50137
      2             2.000000   101400000.000   16.90000        16900
      3             3.000000   197778000.000   32.96300        32963
      4                    *           0.000   0.00000            0

Number of groups with equal values: 3
Sum of non Null cells:
        Volume = 600000000.000
        Percentage = 100.000
        Count = 100000

Sum of all cells:
        Volume = 600000000.000
        Percentage = 100.000
        Count = 100000

```

### Generic example

```sh
#create a small 3D region
g.region n=1000 s=0 e=2000 w=0 t=300 b=0 res3=100

#create a raster3D map with row() and null values
r3.mapcalc "volmap = if(row() == 5, null(), row())"

#run r3.stats with a subrange of 10
r3.stats input=volmap nsteps=10

#the result should look like this
 num   | minimum <= value   | value < maximum    |     volume    |   perc  | cell count
      1          1.000000000          1.900000000    60000000.000   10.00000           60
      2          1.900000000          2.800000000    60000000.000   10.00000           60
      3          2.800000000          3.700000000    60000000.000   10.00000           60
      4          3.700000000          4.600000000    60000000.000   10.00000           60
      5          4.600000000          5.500000000           0.000   0.00000            0
      6          5.500000000          6.400000000    60000000.000   10.00000           60
      7          6.400000000          7.300000000    60000000.000   10.00000           60
      8          7.300000000          8.200000000    60000000.000   10.00000           60
      9          8.200000000          9.100000000    60000000.000   10.00000           60
     10          9.100000000         10.000000001    60000000.000   10.00000           60
     11                    *                    *    60000000.000   10.00000           60

Sum of non Null cells:
        Volume = 540000000.000
        Percentage =  90.000
        Cell count = 540

Sum of all cells:
        Volume = 600000000.000
        Percentage = 100.000
        Cell count = 600
```

## SEE ALSO

*[g.region](g.region.md), [r3.univar](r3.univar.md),
[r.stats](r.stats.md), [r.statistics](r.statistics.md),
[v.univar](v.univar.md)*

## AUTHOR

Sören Gebbert

## SOURCE CODE

Available at: [r3.stats source code](https://github.com/OSGeo/grass/tree/main/raster3d/r3.stats)
([history](https://github.com/OSGeo/grass/commits/main/raster3d/r3.stats))  
Latest change: Friday Feb 07 19:16:09 2025 in commit [a82a39f](https://github.com/OSGeo/grass/commit/a82a39f673b97fafcfce1604dc3e6015cb96487a)
