---
name: r.uslek
description: Computes USLE Soil Erodibility Factor (K).
keywords: [ raster, hydrology, soil, erosion ]
---

# r.uslek

Computes USLE Soil Erodibility Factor (K).

=== "Command line"

    **r.uslek**
    **psand**=*name*
    **pclay**=*name*
    **psilt**=*name*
    **pomat**=*name*
    **output**=*name*
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.uslek psand=name pclay=name psilt=name pomat=name output=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.uslek***",
        **psand**,
        **pclay**,
        **psilt**,
        **pomat**,
        **output**,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.uslek", psand="name", pclay="name", psilt="name", pomat="name", output="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_uslek*(**psand**,
        **pclay**,
        **psilt**,
        **pomat**,
        **output**,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_uslek(psand="name", pclay="name", psilt="name", pomat="name", output="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **psand**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of soil sand fraction raster map [0.0-1.0]  
    **pclay**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of soil clay fraction raster map [0.0-1.0]  
    **psilt**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of soil silt fraction raster map [0.0-1.0]  
    **pomat**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of soil organic matter raster map [0.0-1.0]  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output USLE K factor raster map [t.ha.hr/ha.MJ.mm]  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **psand** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of soil sand fraction raster map [0.0-1.0]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **pclay** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of soil clay fraction raster map [0.0-1.0]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **psilt** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of soil silt fraction raster map [0.0-1.0]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **pomat** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of soil organic matter raster map [0.0-1.0]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output USLE K factor raster map [t.ha.hr/ha.MJ.mm]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **psand** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of soil sand fraction raster map [0.0-1.0]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **pclay** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of soil clay fraction raster map [0.0-1.0]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **psilt** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of soil silt fraction raster map [0.0-1.0]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **pomat** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of soil organic matter raster map [0.0-1.0]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output USLE K factor raster map [t.ha.hr/ha.MJ.mm]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.uslek* calculates the USLE K factor, that is the Soil Erodibility
Factor. It takes input of soil texture classes (sand, clay, silt) and
organic matter, all in range of \[0.0-1.0\]. The FAO World Soil CD
documentation was used to produce the conversion system between soil
textures and soil classes. The soil classes are in number of 12 and
apparently come from a USDA publication of 1951 (p.209). Once the soil
classes have been identified (by vector cross-products tests), a general
conversion table was applied for transforming soil classes into K
factor.

## NOTES

*r.watershed* provides for USLE L, S, LS factors.

## EXAMPLE

Calculate the USLE K factor (synthetic values, North Carolina sample
dataset):

```sh
# NC data: simulation
g.region rural_1m -p

r.mapcalc "sand  = 0.11"
r.mapcalc "clay  = 0.47"
r.mapcalc "silt  = 0.42"
r.mapcalc "organ = 0.0136"

r.uslek psand=sand pclay=clay psilt=silt pomat=organ output=usle_k
r.univar usle_k
# Result: uslek = 0.25
```

Example with small variations (condition: d_sand + d_clay + d_silt =\>
1.0)

```sh
# NC data: simulation
r.mapcalc -s "sand_r = rand(0.09,0.13)"
r.mapcalc -s "clay_r = rand(0.45,0.49)"
r.mapcalc -s "silt_r = rand(0.40,0.44)"
r.mapcalc -s "organ  = rand(0.01,0.70)"

r.mapcalc "temp_sum = sand_r + clay_r + silt_r"
r.mapcalc "sand = sand_r / temp_sum"
r.mapcalc "clay = clay_r / temp_sum"
r.mapcalc "silt = silt_r / temp_sum"
r.mapcalc "test = sand + clay + silt"
r.info -r test
g.remove -f rast name=sand_r,clay_r,silt_r,temp_sum,test

r.uslek psand=sand pclay=clay psilt=silt pomat=organ output=usle_k
r.univar usle_k
# Result: uslek =  variable values
```

## SEE ALSO

*[r.usler](r.usler.md), [r.watershed](r.watershed.md)*

## AUTHOR

Yann Chemin, SIC-ISDC, Turkmenistan

## SOURCE CODE

Available at: [r.uslek source code](https://github.com/OSGeo/grass/tree/main/raster/r.uslek)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.uslek))  
Latest change: Friday Mar 07 07:39:48 2025 in commit [e1e37d8](https://github.com/OSGeo/grass/commit/e1e37d87e78212dfcf4c97aaf32c54042adbd2b0)
