---
name: r.in.bin
description: Import a binary raster file into a GRASS raster map layer.
keywords: [ raster, import ]
---

# r.in.bin

Import a binary raster file into a GRASS raster map layer.

=== "Command line"

    **r.in.bin**
    [**-fdsbh**]
    **input**=*name*
    **output**=*name*
    [**title**=*phrase*]
    [**bytes**=*integer*]
    [**header**=*integer*]
    [**bands**=*integer*]
    [**order**=*string*]
    [**north**=*float*]
    [**south**=*float*]
    [**east**=*float*]
    [**west**=*float*]
    [**rows**=*integer*]
    [**cols**=*integer*]
    [**anull**=*float*]
    [**flip**=*string* [,*string*,...]]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.in.bin input=name output=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.in.bin***",
        **input**,
        **output**,
        **title**=*None*,
        **bytes**=*None*,
        **header**=*0*,
        **bands**=*1*,
        **order**=*"native"*,
        **north**=*None*,
        **south**=*None*,
        **east**=*None*,
        **west**=*None*,
        **rows**=*None*,
        **cols**=*None*,
        **anull**=*None*,
        **flip**=*None*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.in.bin", input="name", output="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_in_bin*(**input**,
        **output**,
        **title**=*None*,
        **bytes**=*None*,
        **header**=*0*,
        **bands**=*1*,
        **order**=*"native"*,
        **north**=*None*,
        **south**=*None*,
        **east**=*None*,
        **west**=*None*,
        **rows**=*None*,
        **cols**=*None*,
        **anull**=*None*,
        **flip**=*None*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_in_bin(input="name", output="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of binary raster file to be imported  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output name or prefix if several bands are imported  
    **title**=*phrase*  
    &nbsp;&nbsp;&nbsp;&nbsp;Title for resultant raster map  
    **bytes**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of bytes per cell  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1, 2, 4, 8*  
    **header**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Header size in bytes  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **bands**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of bands in input file  
    &nbsp;&nbsp;&nbsp;&nbsp;Bands must be in band-sequential order  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **order**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output byte order  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *big, little, native, swap*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *native*  
    **north**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Northern limit of geographic region (outer edge)  
    **south**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Southern limit of geographic region (outer edge)  
    **east**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Eastern limit of geographic region (outer edge)  
    **west**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Western limit of geographic region (outer edge)  
    **rows**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of rows  
    **cols**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of columns  
    **anull**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Set Value to NULL  
    **flip**=*string* [,*string*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Flip input horizontal and/or vertical  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *h, v*  
    &nbsp;&nbsp;&nbsp;&nbsp;**h**: Flip input horizontal (East - West)  
    &nbsp;&nbsp;&nbsp;&nbsp;**v**: Flip input vertical (North - South)  
    **-f**  
    &nbsp;&nbsp;&nbsp;&nbsp;Import as floating-point data (default: integer)  
    **-d**  
    &nbsp;&nbsp;&nbsp;&nbsp;Import as double-precision floating-point data (default: integer)  
    **-s**  
    &nbsp;&nbsp;&nbsp;&nbsp;Signed data (two's complement)  
    **-b**  
    &nbsp;&nbsp;&nbsp;&nbsp;Byte swap the data during import  
    **-h**  
    &nbsp;&nbsp;&nbsp;&nbsp;Get region info from GMT style header  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of binary raster file to be imported  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output name or prefix if several bands are imported  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **title** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Title for resultant raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *phrase*  
    **bytes** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of bytes per cell  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1, 2, 4, 8*  
    **header** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Header size in bytes  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **bands** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of bands in input file  
    &nbsp;&nbsp;&nbsp;&nbsp;Bands must be in band-sequential order  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **order** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output byte order  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *big, little, native, swap*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *native*  
    **north** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Northern limit of geographic region (outer edge)  
    **south** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Southern limit of geographic region (outer edge)  
    **east** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Eastern limit of geographic region (outer edge)  
    **west** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Western limit of geographic region (outer edge)  
    **rows** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of rows  
    **cols** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of columns  
    **anull** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Set Value to NULL  
    **flip** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Flip input horizontal and/or vertical  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *h, v*  
    &nbsp;&nbsp;&nbsp;&nbsp;**h**: Flip input horizontal (East - West)  
    &nbsp;&nbsp;&nbsp;&nbsp;**v**: Flip input vertical (North - South)  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *f*, *d*, *s*, *b*, *h*  
    &nbsp;&nbsp;&nbsp;&nbsp;**f**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Import as floating-point data (default: integer)  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Import as double-precision floating-point data (default: integer)  
    &nbsp;&nbsp;&nbsp;&nbsp;**s**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Signed data (two's complement)  
    &nbsp;&nbsp;&nbsp;&nbsp;**b**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Byte swap the data during import  
    &nbsp;&nbsp;&nbsp;&nbsp;**h**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Get region info from GMT style header  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of binary raster file to be imported  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output name or prefix if several bands are imported  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **title** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Title for resultant raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *phrase*  
    **bytes** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of bytes per cell  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1, 2, 4, 8*  
    **header** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Header size in bytes  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **bands** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of bands in input file  
    &nbsp;&nbsp;&nbsp;&nbsp;Bands must be in band-sequential order  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **order** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output byte order  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *big, little, native, swap*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *native*  
    **north** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Northern limit of geographic region (outer edge)  
    **south** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Southern limit of geographic region (outer edge)  
    **east** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Eastern limit of geographic region (outer edge)  
    **west** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Western limit of geographic region (outer edge)  
    **rows** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of rows  
    **cols** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of columns  
    **anull** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Set Value to NULL  
    **flip** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Flip input horizontal and/or vertical  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *h, v*  
    &nbsp;&nbsp;&nbsp;&nbsp;**h**: Flip input horizontal (East - West)  
    &nbsp;&nbsp;&nbsp;&nbsp;**v**: Flip input vertical (North - South)  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *f*, *d*, *s*, *b*, *h*  
    &nbsp;&nbsp;&nbsp;&nbsp;**f**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Import as floating-point data (default: integer)  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Import as double-precision floating-point data (default: integer)  
    &nbsp;&nbsp;&nbsp;&nbsp;**s**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Signed data (two's complement)  
    &nbsp;&nbsp;&nbsp;&nbsp;**b**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Byte swap the data during import  
    &nbsp;&nbsp;&nbsp;&nbsp;**h**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Get region info from GMT style header  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.in.bin* allows the user to create a (binary) GRASS raster map layer
from a variety of binary raster data formats.

The **-s** flag is used for importing two's-complement signed data.

The **-h** flag is used to read region information from a Generic
Mapping Tools (GMT) type binary header. It is compatible with GMT binary
grid types 1 and 2.

The north, south, east, and west field values are the coordinates of the
edges of the geographic region. The rows and cols values describe the
dimensions of the matrix of data to follow. If the input is a
[GMT](https://www.generic-mapping-tools.org/) binary array (-h flag),
the six dimension fields (north, south, east, west, rows and cols) are
obtained from the GMT header. If the bytes field is entered incorrectly
an error will be generated suggesting a closer bytes value.

*r.in.bin* can be used to import numerous binary arrays including:
ETOPO30, ETOPO-5, ETOPO-2, Globe DEM, BIL, AVHRR and GMT binary arrays
(ID 1 & 2).

## NOTES

If optional parameters are not supplied, **r.in.bin** attempts to
calculate them. For example if the rows and columns parameters are not
entered, **r.in.bin** automatically calculates them by subtracting south
from north and west from east. This will only produce correct results if
the raster resolution equals 1. Also, if the north, south, east, and
west parameters are not entered, **r.in.bin** assigns them from the rows
and columns parameters. In the AVHRR example (see below), the raster
would be assigned a north=128, south=0, east=128, west=0.

The geographic coordinates north, south, east, and west describe the
outer edges of the geographic region. They run along the edges of the
cells at the edge of the geographic region and *not* through the center
of the cells at the edges.

Eastern limit of geographic region (in projected coordinates must be
east of the west parameter value, but in geographical coordinates will
wrap around the globe; user errors can be detected by comparing the
*ewres* and *nsres* values of the imported map layer carefully).  
Western limit of geographic region (in projected coordinates must be
west of the east parameter value, but in geographical coordinates will
wrap around the globe; user errors can be detected by comparing the
*ewres* and *nsres* values of the imported map layer carefully).

Notes on (non)signed data:

If you use the **-s** flag, the highest bit is the sign bit. If this is
1, the data is negative, and the data interval is half of the unsigned
(not exactly).

This flag is only used if **bytes=** 1. If **bytes** is greater than 1,
the flag is ignored.

## EXAMPLES

### GTOPO30 DEM

The following is a sample call of *r.in.bin* to import [GTOPO30
DEM](http://edcdaac.usgs.gov/gtopo30/gtopo30.asp) data:

```sh
r.in.bin -sb input=E020N90.DEM output=gtopo30 bytes=2 north=90 south=40
east=60 west=20 r=6000 c=4800
```

(you can add "anull=-9999" if you want sea level to have a NULL value)

### GMT

The following is a sample call of *r.in.bin* to import a GMT type 1
(float) binary array:

```sh
r.in.bin -hf input=sample.grd output=sample.grass
```

(-b could be used to swap bytes if required)

### AVHRR

The following is a sample call of *r.in.bin* to import an AVHRR image:

```sh
r.in.bin in=p07_b6.dat out=avhrr c=128 r=128
```

### ETOPO2

The following is a sample call of *r.in.bin* to import [ETOPO2
DEM](http://www.ngdc.noaa.gov/mgg/image/2minrelief.html) data (here full
data set):

```sh
r.in.bin ETOPO2.dos.bin out=ETOPO2min r=5400 c=10800 n=90 s=-90 w=-180 e=180 bytes=2
r.colors ETOPO2min rules=terrain
```

### TOPEX/SRTM30 PLUS

The following is a sample call of *r.in.bin* to import [SRTM30
PLUS](http://topex.ucsd.edu/WWW_html/srtm30_plus.html) data:

```sh
r.in.bin -sb input=e020n40.Bathymetry.srtm output=e020n40_topex \
         bytes=2 north=40 south=-10 east=60 west=20 r=6000 c=4800
r.colors e020n40_topex rules=etopo2
```

### GPCP

The following is a sample call of *r.in.bin* to import GPCP 1DD v1.2
data:

```sh
YEAR="2000"
MONTH="01"
# number of days of this month
MDAYS=`date -d"${YEAR}-${MONTH}-01 + 1 month - 1 day" +%d`
r.in.bin in=gpcp_1dd_v1.2_p1d.${YEAR}${MONTH} out=gpcp_${YEAR}.${MONTH}. \
         order=big bytes=4 -f header=1440 anull=-99999 \
         n=90 s=-90 w=0 e=360 rows=180 cols=360 bands=$MDAYS
```

The following is a sample call of *r.in.bin* to import GPCP v2.2 data:

```sh
r.in.bin in=gpcp_v2.2_psg.1979 out=gpcp_1979. \
         order=big bytes=4 -f header=576 anull=-99999 \
         n=90 s=-90 w=0 e=360 rows=72 cols=144 bands=12
```

## SEE ALSO

*[r.import](r.import.md), [r.out.bin](r.out.bin.md),
[r.in.ascii](r.in.ascii.md), [r.out.ascii](r.out.ascii.md),
[r.in.gdal](r.in.gdal.md), [r.out.gdal](r.out.gdal.md),
[r.in.srtm](r.in.srtm.md)*

## AUTHORS

Jacques Bouchard, France (<bouchard@onera.fr>)  
Bob Covill, Canada (<bcovill@tekmap.ns.ca>)  
Markus Metz  
Man page: Zsolt Felker (<felker@c160.pki.matav.hu>)

## SOURCE CODE

Available at: [r.in.bin source code](https://github.com/OSGeo/grass/tree/main/raster/r.in.bin)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.in.bin))  
Latest change: Thursday May 08 20:58:05 2025 in commit [4144ba8](https://github.com/OSGeo/grass/commit/4144ba81fa125e1549fc52e1cf5feb4fa004be8b)
